########################################################################################################################################
########################################################################################################################################
# LEBubleyDyer
########################################################################################################################################
########################################################################################################################################

#' @title
#'Generator of linear extensions through the Bubley-Dyer procedure.
#'
#' @description
#' Creates an object of S4 class `LEBubleyDyer`, needed to sample the linear extensions of a given poset according to the Bubley-Dyer
#' procedure. Actually, this function does not sample the linear extensions, but just generates the object that will sample them
#' by using function `LEGet`.
#'
#' @param poset An object of S4 class `POSet` representing the poset whose linear extensions are generated.
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .
#'
#' @param seed Positive integer to initialize the random linear extension generation.
#'
#' @return
#' An object of class `LEBubleyDyer`.
#'
#'
#' @examples
#' el <- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "b", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = el, dom = dom)
#'
#' LEgenBD <- LEBubleyDyer(pos)
#'
#' @name LEBubleyDyer
#' @export LEBubleyDyer
LEBubleyDyer <- function(poset, seed=NULL) {
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  if (is.null(seed)) {
    args <- list()
  } else {
    if (seed < 0 || seed != round(seed)) {
      stop("seed must be a positive integer")
    }
    args <- list(BubleyDyerSeed=seed)
  }
  tryCatch({
    ptr <- .Call("_BuildBubleyDyerLEGenerator", poset@ptr, args)
    result <- methods::new("BubleyDyerGenerator", ptr=ptr)
    return (result)
  }, error = function(err) {
    err_split <- strsplit(err[[1]], split = ":")
    stop(err_split[[1]][length(err_split[[1]])])
  }) # END tryCatch
}
