% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linkGRASS.R
\name{linkGRASS}
\alias{linkGRASS}
\title{Locate and set up GRASS GIS API bindings}
\usage{
linkGRASS(
  x = NULL,
  epsg = NULL,
  default_GRASS = NULL,
  search_path = NULL,
  ver_select = FALSE,
  gisdbase_exist = FALSE,
  gisdbase = NULL,
  use_home = FALSE,
  location = NULL,
  spatial_params = NULL,
  resolution = NULL,
  quiet = TRUE,
  returnPaths = TRUE
)
}
\arguments{
\item{x}{A spatial object used to initialize the GRASS location.
Supported classes are `terra::SpatRaster`, `sf`, `sp`, `stars`,
or a file path to a raster dataset.}

\item{epsg}{Integer EPSG code used to define the GRASS projection.
If `NULL`, the EPSG code is inferred from `x` when possible.}

\item{default_GRASS}{Optional character vector defining a GRASS installation
(e.g. `c("/usr/lib/grass83", "8.3.2", "grass")`).}

\item{search_path}{Character path used to search for GRASS installations.}

\item{ver_select}{Logical or numeric value controlling interactive or indexed
GRASS version selection.}

\item{gisdbase_exist}{Logical; if `TRUE`, `gisdbase` and `location` must
already exist and will only be linked.}

\item{gisdbase}{Path to the GRASS database directory.}

\item{use_home}{Logical; if `TRUE`, the user home directory is used for GISRC.}

\item{location}{Name of the GRASS location to create or link.}

\item{spatial_params}{Optional numeric vector defining extent manually
(`xmin, ymin, xmax, ymax[, proj]`).}

\item{resolution}{Numeric raster resolution used for `g.region`.}

\item{quiet}{Logical; suppress console output if `TRUE`.}

\item{returnPaths}{Logical; return detected GRASS installation paths.}
}
\value{
A list describing the selected GRASS installation and status, or `NULL`
if no valid installation was found.
}
\description{
Initializes a GRASS GIS 7.x/8.x runtime environment and prepares a valid
temporary or permanent GRASS location and mapset for use from R.
}
\details{
The function detects installed GRASS versions, initializes required
environment variables, and derives spatial reference information either
from an existing spatial object or from manually provided parameters.


GRASS requires a fully initialized runtime environment (PATH, GISBASE,
PROJ, GDAL). On some platforms, R must be started from a shell where GRASS
is already available.

The function ensures that `PROJ_INFO` and `PROJ_UNITS` are written by
explicitly calling `g.proj` before region initialization.
}
\examples{
\dontrun{
library(link2GI)
library(sf)

# Example 1: initialize a temporary GRASS location from an sf object
nc <- st_read(system.file("shape/nc.shp", package = "sf"))
grass <- linkGRASS(nc)

# Example 2: select GRASS version interactively if multiple installations exist
linkGRASS(nc, ver_select = TRUE)

# Example 3: create a permanent GRASS location
root <- tempdir()
linkGRASS(
  x        = nc,
  gisdbase = root,
  location = "project1"
)

# Example 4: link to an existing GRASS location without recreating it
linkGRASS(
  gisdbase        = root,
  location        = "project1",
  gisdbase_exist  = TRUE
)

# Example 5: manual setup using spatial parameters only
epsg <- 28992
linkGRASS(
  spatial_params = c(178605, 329714, 181390, 333611),
  epsg = epsg
)
}
}
\seealso{
\code{\link[rgrass]{initGRASS}},
\code{\link[rgrass]{execGRASS}}
}
\author{
Chris Reudenbach
}
