/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5EApkg.h"     
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5VMprivate.h" 

H5FL_DEFINE_STATIC(H5EA_iblock_t);

H5FL_BLK_DEFINE_STATIC(idx_blk_elmt_buf);

H5FL_SEQ_DEFINE_STATIC(haddr_t);

H5EA_iblock_t *
H5EA__iblock_alloc(H5EA_hdr_t *hdr)
{
    H5EA_iblock_t *iblock    = NULL; 
    H5EA_iblock_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (NULL == (iblock = H5FL_CALLOC(H5EA_iblock_t)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "memory allocation failed for extensible array index block");

    
    if (H5EA__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINC, NULL, "can't increment reference count on shared array header");
    iblock->hdr = hdr;

    
    iblock->addr = HADDR_UNDEF;

    
    iblock->nsblks      = H5EA_SBLK_FIRST_IDX(hdr->cparam.sup_blk_min_data_ptrs);
    iblock->ndblk_addrs = 2 * ((size_t)hdr->cparam.sup_blk_min_data_ptrs - 1);
    iblock->nsblk_addrs = hdr->nsblks - iblock->nsblks;

    
    if (hdr->cparam.idx_blk_elmts > 0)
        if (NULL ==
            (iblock->elmts = H5FL_BLK_MALLOC(
                 idx_blk_elmt_buf, (size_t)(hdr->cparam.idx_blk_elmts * hdr->cparam.cls->nat_elmt_size))))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                        "memory allocation failed for index block data element buffer");

    
    if (iblock->ndblk_addrs > 0)
        if (NULL == (iblock->dblk_addrs = H5FL_SEQ_MALLOC(haddr_t, iblock->ndblk_addrs)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                        "memory allocation failed for index block data block addresses");

    
    if (iblock->nsblk_addrs > 0)
        if (NULL == (iblock->sblk_addrs = H5FL_SEQ_MALLOC(haddr_t, iblock->nsblk_addrs)))
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                        "memory allocation failed for index block super block addresses");

    
    ret_value = iblock;

done:
    if (!ret_value)
        if (iblock && H5EA__iblock_dest(iblock) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, NULL, "unable to destroy extensible array index block");
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5EA__iblock_create(H5EA_hdr_t *hdr, bool *stats_changed)
{
    H5EA_iblock_t *iblock = NULL;     
    haddr_t        iblock_addr;       
    bool           inserted  = false; 
    haddr_t        ret_value = HADDR_UNDEF;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(stats_changed);

    
    if (NULL == (iblock = H5EA__iblock_alloc(hdr)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "memory allocation failed for extensible array index block");

    
    iblock->size = H5EA_IBLOCK_SIZE(iblock);

    
    if (HADDR_UNDEF == (iblock_addr = H5MF_alloc(hdr->f, H5FD_MEM_EARRAY_IBLOCK, (hsize_t)iblock->size)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, HADDR_UNDEF,
                    "file allocation failed for extensible array index block");
    iblock->addr = iblock_addr;

    
    if (hdr->cparam.idx_blk_elmts > 0) {
        
        if ((hdr->cparam.cls->fill)(iblock->elmts, (size_t)hdr->cparam.idx_blk_elmts) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "can't set extensible array index block elements to class's fill value");
    } 

    
    if (iblock->ndblk_addrs > 0) {
        haddr_t tmp_addr = HADDR_UNDEF; 

        
        H5VM_array_fill(iblock->dblk_addrs, &tmp_addr, sizeof(haddr_t), iblock->ndblk_addrs);
    } 

    
    if (iblock->nsblk_addrs > 0) {
        haddr_t tmp_addr = HADDR_UNDEF; 

        
        H5VM_array_fill(iblock->sblk_addrs, &tmp_addr, sizeof(haddr_t), iblock->nsblk_addrs);
    } 

    
    if (H5AC_insert_entry(hdr->f, H5AC_EARRAY_IBLOCK, iblock_addr, iblock, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTINSERT, HADDR_UNDEF,
                    "can't add extensible array index block to cache");
    inserted = true;

    
    if (hdr->top_proxy) {
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, iblock) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, HADDR_UNDEF,
                        "unable to add extensible array entry as child of array proxy");
        iblock->top_proxy = hdr->top_proxy;
    } 

    
    assert(0 == hdr->stats.computed.nindex_blks);
    assert(0 == hdr->stats.computed.index_blk_size);
    hdr->stats.computed.nindex_blks    = 1;
    hdr->stats.computed.index_blk_size = iblock->size;

    
    hdr->stats.stored.nelmts += hdr->cparam.idx_blk_elmts;

    
    *stats_changed = true;

    
    ret_value = iblock_addr;

done:
    if (!H5_addr_defined(ret_value))
        if (iblock) {
            
            if (inserted)
                if (H5AC_remove_entry(iblock) < 0)
                    HDONE_ERROR(H5E_EARRAY, H5E_CANTREMOVE, HADDR_UNDEF,
                                "unable to remove extensible array index block from cache");

            
            if (H5_addr_defined(iblock->addr) &&
                H5MF_xfree(hdr->f, H5FD_MEM_EARRAY_IBLOCK, iblock->addr, (hsize_t)iblock->size) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to release file space for extensible array index block");

            
            if (H5EA__iblock_dest(iblock) < 0)
                HDONE_ERROR(H5E_EARRAY, H5E_CANTFREE, HADDR_UNDEF,
                            "unable to destroy extensible array index block");
        } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

H5EA_iblock_t *
H5EA__iblock_protect(H5EA_hdr_t *hdr, unsigned flags)
{
    H5EA_iblock_t *iblock    = NULL; 
    H5EA_iblock_t *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    if (NULL ==
        (iblock = (H5EA_iblock_t *)H5AC_protect(hdr->f, H5AC_EARRAY_IBLOCK, hdr->idx_blk_addr, hdr, flags)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, NULL,
                    "unable to protect extensible array index block, address = %llu",
                    (unsigned long long)hdr->idx_blk_addr);

    
    if (hdr->top_proxy && NULL == iblock->top_proxy) {
        
        if (H5AC_proxy_entry_add_child(hdr->top_proxy, hdr->f, iblock) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTSET, NULL,
                        "unable to add extensible array entry as child of array proxy");
        iblock->top_proxy = hdr->top_proxy;
    } 

    
    ret_value = iblock;

done:
    
    if (!ret_value) {
        
        if (iblock &&
            H5AC_unprotect(hdr->f, H5AC_EARRAY_IBLOCK, iblock->addr, iblock, H5AC__NO_FLAGS_SET) < 0)
            HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, NULL,
                        "unable to unprotect extensible array index block, address = %llu",
                        (unsigned long long)iblock->addr);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__iblock_unprotect(H5EA_iblock_t *iblock, unsigned cache_flags)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(iblock);

    
    if (H5AC_unprotect(iblock->hdr->f, H5AC_EARRAY_IBLOCK, iblock->addr, iblock, cache_flags) < 0)
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL,
                    "unable to unprotect extensible array index block, address = %llu",
                    (unsigned long long)iblock->addr);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__iblock_delete(H5EA_hdr_t *hdr)
{
    H5EA_iblock_t *iblock    = NULL; 
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(hdr->idx_blk_addr));

    
    if (NULL == (iblock = H5EA__iblock_protect(hdr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTPROTECT, FAIL,
                    "unable to protect extensible array index block, address = %llu",
                    (unsigned long long)hdr->idx_blk_addr);

    
    if (iblock->ndblk_addrs > 0) {
        unsigned sblk_idx; 
        unsigned dblk_idx; 
        size_t   u;        

        
        sblk_idx = dblk_idx = 0;
        for (u = 0; u < iblock->ndblk_addrs; u++) {
            
            if (H5_addr_defined(iblock->dblk_addrs[u])) {
                
                if (H5EA__dblock_delete(hdr, iblock, iblock->dblk_addrs[u],
                                        hdr->sblk_info[sblk_idx].dblk_nelmts) < 0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTDELETE, FAIL,
                                "unable to delete extensible array data block");
                iblock->dblk_addrs[u] = HADDR_UNDEF;
            } 

            
            dblk_idx++;

            
            if (dblk_idx >= hdr->sblk_info[sblk_idx].ndblks) {
                sblk_idx++;
                dblk_idx = 0;
            } 
        }     
    }         

    
    if (iblock->nsblk_addrs > 0) {
        size_t u; 

        
        for (u = 0; u < iblock->nsblk_addrs; u++) {
            
            if (H5_addr_defined(iblock->sblk_addrs[u])) {
                
                if (H5EA__sblock_delete(hdr, iblock, iblock->sblk_addrs[u], (unsigned)(u + iblock->nsblks)) <
                    0)
                    HGOTO_ERROR(H5E_EARRAY, H5E_CANTDELETE, FAIL,
                                "unable to delete extensible array super block");
                iblock->sblk_addrs[u] = HADDR_UNDEF;
            }
        }
    }

done:
    
    if (iblock && H5EA__iblock_unprotect(iblock, H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG |
                                                     H5AC__FREE_FILE_SPACE_FLAG) < 0)
        HDONE_ERROR(H5E_EARRAY, H5E_CANTUNPROTECT, FAIL, "unable to release extensible array index block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5EA__iblock_dest(H5EA_iblock_t *iblock)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(iblock);

    
    if (iblock->hdr) {
        
        if (iblock->elmts) {
            
            assert(iblock->hdr->cparam.idx_blk_elmts > 0);
            iblock->elmts = H5FL_BLK_FREE(idx_blk_elmt_buf, iblock->elmts);
        } 

        
        if (iblock->dblk_addrs) {
            
            assert(iblock->ndblk_addrs > 0);
            iblock->dblk_addrs  = H5FL_SEQ_FREE(haddr_t, iblock->dblk_addrs);
            iblock->ndblk_addrs = 0;
        } 

        
        if (iblock->sblk_addrs) {
            
            assert(iblock->nsblk_addrs > 0);
            iblock->sblk_addrs  = H5FL_SEQ_FREE(haddr_t, iblock->sblk_addrs);
            iblock->nsblk_addrs = 0;
        } 

        
        if (H5EA__hdr_decr(iblock->hdr) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared array header");
        iblock->hdr = NULL;
    } 

    
    assert(NULL == iblock->top_proxy);

    
    iblock = H5FL_FREE(H5EA_iblock_t, iblock);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
