/*
  Author: Xuye Luo
  Date: December 12, 2025
*/

#include <Rcpp.h>
#include <cmath>
using namespace Rcpp;

// [[Rcpp::interfaces(r, cpp)]]


/* 
//' @title Modified Pearson's Chi-squared Statistic (C++ Backend)
//' @description Calculates the Chi-squared statistic ignoring cells with 
//' zero expected values (No-NA approach).
//' @param x NumericMatrix representing the contingency table.
//' @return The calculated statistic.
 */

// [[Rcpp::export]]
double modified_chisq_statistic_cpp(const NumericMatrix &x) {
  
  double N = sum(x);
  
  // Handle empty table case
  if (N == 0) {
    return 0.0;
  }

  int nr = x.nrow();
  int nc = x.ncol();
  
  // Calculate marginal sums
  NumericVector row_sum = rowSums(x);
  NumericVector col_sum = colSums(x);
  
  double statistic = 0.0;
  double inv_N = 1.0 / N; 


  for (int j = 0; j < nc; ++j) {
    double c_sum = col_sum[j];
    

    if (c_sum == 0) continue;

    for (int i = 0; i < nr; ++i) {
      double r_sum = row_sum[i];
      
      // Optimization: Skip row if sum is 0
      if (r_sum == 0) continue;
      
      // Calculate Expected value: (RowSum * ColSum) / N
      double E = r_sum * c_sum * inv_N;
      
      if (E > 0) {
        double O = x(i, j);
        double diff = O - E;
        
        statistic += (diff * diff) / E;
      }
    }
  }
  
  return statistic;
}
