#' @title Solidarity value
#' @description Given a game, this function computes its solidarity value.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param amc A logical value. By default, \code{amc=FALSE}. If set to \code{TRUE}, the average marginal contributions are also returned.
#' @return The solidarity value of the game, as a vector. If \code{amc=TRUE}, a vector (in binary order if \code{binary=TRUE} and in lexicographic order otherwise) containing the average marginal contribution to each coalition is also returned.
#' @details Given \eqn{v\in G^N}, the average marginal contribution to coalition \eqn{S\in 2^N} is defined as
#' \deqn{AMC(S)=\frac{1}{|S|}\sum_{k\in S}(v(S)-v(S\backslash \{k\})).}
#' The solidarity value of each player \eqn{i \in N} can be defined as
#' \deqn{\phi_i(v)=\sum_{S : i\in S}\frac{(n-|S|)!(|S|-1)!}{|N|!}AMC(S).}
#' @examples
#' solidarityvalue(c(0,0,0,1,1,1,2), binary=TRUE)
#' solidarityvalue(bin2lex(c(0,0,1,2,5,5,7)))
#' solidarityvalue(bin2lex(c(0,0,2,7,9,10,12,9,11,12,14,19,21,22,24)), amc=TRUE)
#' @references Nowak, A. S. & Radzik, T. (1994). A solidarity value for n-person transferable utility games. \emph{International Journal of Game Theory}, 23, 43-48.
#' @seealso \link{shapleyvalue}
#' @export

solidarityvalue <- function(v, binary = FALSE, amc = FALSE) {
  # SOLIDARITY VALUE
  # solidarityvalue(v) Computes the solidarity value for a game v
  # v: characteristic function (binary order)
  # Output: solv: solidarity value of v
          #mcm: average marginal contribution vector
  #Examples:
  #         solidarityvalue(c(0,0,0,1,1,1,2))
  #         returns [0.5278    0.5278    0.9444]
  #         solidarityvalue(c(0, 0, 1, 2, 5, 5, 7))
  #         returns [1.9444    1.9444    3.1111]
  #         solidarityvalue(c(0,0, 2, 7, 9, 10,12, 9, 11, 12, 14, 19, 21, 22, 24))
  #         returns [4.4097    4.6042    7.1319    7.8542]
  # v <- c(0,0,1,0,0,0,1)
  # solidarityvalue(v)


  nC <- length(v)
  n <- log2(nC + 1)
  if (n>floor(n))
  {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  # Cálculo media contribuciones marginales:  A(S), aquí mcm(S)
  mcm <- v  # Inicializo mcm cos v iniciais, porque non hai que cambiar os individuais e no resto haberá que facer V(S)-algo

  for (S in 1:(2^n - 1)) {
    coalition <- as.integer(intToBits(S))[1:n] # Xogadores en S e cardinal
    scoal <- sum(coalition)
    pos <- which(coalition > 0) #  Localizo os xogadores

    if (scoal > 1) { # Só fago cambios en mcm se hai máis de un xogador en S
      mcm_aux <- 0
      for (kk in seq_along(pos)) { # Vou eliminando cada un dos xogadores de S
        expo <- pos
        expo <- expo[-kk]  # remove player kk
        idx <- sum(2^(expo - 1))
        mcm_aux <- mcm_aux + v[idx] #Calculamos V(S\kk)
      }
      mcm[S] <- mcm[S] - mcm_aux / scoal # Cálculo final
    }
  }

  ##############Valores que aparecen na matriz de Shapley ##########
  ###### Aquí mesmo código que temos para o cálculo da matriz de Shapley #########
  metade <- floor(n / 2 + 0.5)
  valores <- rep(1/n, metade)

  if (metade >= 2) {
    for (ii in 2:metade) {
      valores[ii] <- valores[ii - 1] * (ii - 1) / (n - ii + 1)
    }
  }

  if (n %% 2 == 0) {
    vshap <- c(valores, sort(valores))
  } else {
    vshap <- c(valores, sort(valores[-length(valores)]))
  }


  ############ Cálculo do valor #############
  #### #Aquí é onde vou empregar os valores da matriz de Shapley, pero SEMPRE
  ### AQUÍ VAN CON SIGNO POSITIVO!!

  solv <- numeric(n)

  for (ii in 1:(2^(n - 1) - 1)) {
    coalition <- as.integer(intToBits(ii))[1:n]
    a <- sum(coalition)
    solv <- solv + coalition * mcm[ii] * vshap[a] + (1 - coalition) * mcm[nC - ii] * vshap[a + 1]
  }

  solv <- solv + mcm[length(mcm)] / n  # Engadimos a parte correspontente a v(N)

  ###############
  ### salidas ###
  ###############

  if (amc == FALSE) {
    return(solv)
  } else {
    if (binary == FALSE) {
      mcm <- bin2lex(mcm)
    }
    return(list(solv = solv, amc = mcm))
  }

}
