#' @importFrom geometry convhulln
#' @importFrom plotly add_trace
#' @importFrom plotly layout
#' @importFrom plotly plot_ly
#' @importFrom stats na.omit
#' @importFrom stringr %>%

polygonal <- function(C, P = NULL) {
  #  C is a nx2 o nx3 matrix whose rows are points that belong to the
  #  same hyperplane.
  #  P must be a 1xn matrix of integer numbers.
  #  CO=polygonal(C) is obtained by ordering the rows of
  #  C so that it is ready to be used  to draw a polygon.

  # polygonal(C,P) if P is a vector that represents the FACES to
  # where the rows of C belong, PO reorders the faces.
  # By default P=[1:n].
  #
  # C <- matrix(c(0, 1, 2/3, -1, 3, -1/3, 1, 0, 1, 7, 7, -17/3, 2, 2, -2/3), ncol = 3, byrow = TRUE)
  # polygonal(C,P=c(5,8,3,2,7))

  #library(pracma), tiene el producto vectorial, añado la definición arriba para evitar pracma

  # Eliminamos filas repetidas
  C <- unique(C)
  n <- nrow(C)

  if (is.null(P) || length(P) < n) {
    P <- 1:n
  } else {
    P <- abs(round(P[1:n]))
  }

  # Comprobamos que los vectores tienen 2 o 3 coordenadas
  if (ncol(C) < 2 || ncol(C) > 3) {
    stop("Matrix 'C' must have 2 or 3 columns.")
  }

  # Si hay menos de 4 puntos, devolvemos sin ordenación
  if (n < 4) {
    return(list(CO = C, PO = P))
  }

  # Llamamos a la función adecuada según la dimensión
  if (ncol(C) == 2) {
    return(polygonal2(C, P))
  } else {
    return(polygonal3(C, P))
  }
}

polygonal2 <- function(C, P) {
  n <- nrow(C)

  # Encontramos el vector menor en orden lexicográfico
  min_x <- min(C[, 1])
  indices <- which(C[, 1] == min_x)

  if (length(indices) == 1) {
    min_index <- indices
  } else {
    min_index <- indices[which.min(C[indices, 2])]
  }

  # Reordenamos
  C[c(1, min_index), ] <- C[c(min_index, 1), ]
  P[c(1, min_index)] <- P[c(min_index, 1)]

  # Calculamos los ángulos con respecto a la dirección de referencia
  ref_dir <- c(C[1, 1], C[1, 2] - 1) - C[1, ]

  angles <- sapply(2:n, function(i) {
    vec <- C[i, ] - C[1, ]
    acos(sum(ref_dir * vec) / (sqrt(sum(ref_dir^2)) * sqrt(sum(vec^2))))
  })

  # Ordenamos los puntos por ángulo
  order_indices <- order(angles)
  C <- rbind(C[1, ], C[order_indices + 1, ])
  P <- c(P[1], P[order_indices + 1])

  return(list(CO = C, PO = P))
}

polygonal3 <- function(C, P) {
  n <- nrow(C)

  # Verificamos que los vectores son coplanarios
  # normal <- crossprod(matrix(C[2, ] - C[1, ], ncol = 3), matrix(C[3, ] - C[1, ], ncol = 3))
  normal <- prodvect(C[2, ] - C[1, ], C[3, ] - C[1, ])
  for (i in 4:n) {
    vec <- C[i, ] - C[1, ]
    if (abs(sum(normal * vec)) > 1e-6) {
      stop("Row vectors in 'C' do not belong to the same plane.")
    }
  }

  # Determinamos la proyección y ordenamos
  if (sum(normal * c(0, 0, 1)) != 0) {
    proj <- C[, 1:2]
  } else if (sum(normal * c(0, 1, 0)) == 0) {
    proj <- C[, 2:3]
  } else {
    proj <- C[, c(1, 3)]
  }

  ordered <- polygonal2(proj, 1:n)

  C <- C[ordered$PO, ]
  P <- P[ordered$PO]

  return(list(CO = C, PO = P))
}

prodvect <- function(a, b) {
  return(c(a[2] * b[3] - a[3] * b[2],
           a[3] * b[1] - a[1] * b[3],
           a[1] * b[2] - a[2] * b[1]))
}

projectionmatrix4 <- function(v) {
  # Verificar que v es un vector de longitud 15
  if (length(v) != 15) {
    stop("'A' must be a 1x15 vector.")
  }

  # Asignar valores de v a variables individuales
  v1 <- v[1]; v2 <- v[2]; v3 <- v[3]; v4 <- v[4]
  v12 <- v[5]; v13 <- v[6]; v14 <- v[7]; v23 <- v[8]; v24 <- v[9]; v34 <- v[10]
  v123 <- v[11]; v124 <- v[12]; v134 <- v[13]; v234 <- v[14]
  v1234 <- v[15]

  # Longitud L del lado del tetraedro de imputaciones
  ll <- v1234 - v1 - v2 - v3 - v4

  if (ll < 0) {
    warning("Non essential game")
    return(NULL)
  } else if (ll == 0) {
    warning(sprintf("The imputation set is the point (%f, %f, %f)", v1, v2, v3))
    return(NULL)
  }

  # Construcción de la matriz de vértices en R4
  verticesreales <- matrix(
    c(v1234 - v2 - v3 - v4, v2, v3, v4,
      v1, v1234 - v1 - v3 - v4, v3, v4,
      v1, v2, v1234 - v1 - v2 - v4, v4,
      v1, v2, v3, v1234 - v1 - v2 - v3),
    nrow = 4, byrow = TRUE
  )

  # Matriz de cambio de base de la canónica a B
  McambioCaB <- solve(t(verticesreales))

  # Matriz de proyección en R3
  Mproyeccion <- matrix(
    c(0, ll, ll/2, ll/2,
      0, 0, sqrt(3)/2 * ll, sqrt(3)/6 * ll,
      0, 0, 0, sqrt(6)/3 * ll),
    nrow = 3, byrow = TRUE
  )

  # Matriz de transformación final
  M <- Mproyeccion %*% McambioCaB

  return(M)
}

################################################################################
### plot2real ##################################################################
################################################################################

plot2real <- function(v,imputations=TRUE,solutions=NULL,allocations=NULL,color="blue") {

  if(essentialcheck(v)==FALSE){
    message("The set of imputations of 'v' is empty (and so is the core).")
    return(NULL)
  }
  v_str <- paste("[", paste(v, collapse = ","), "]", sep = "")
  points <- data.frame(x=corevertices234(v)[,1],y=corevertices234(v)[,2])

  # Fabricamos la gráfica y la ajustamos
  fig <- plot_ly() %>%
    layout(
      title = list(text=paste("core of",v_str),font=list(color=color)),
      xaxis = list(title = "Player 1"),
      yaxis = list(title = "Player 2", scaleanchor = "x"), # Lock Y axis to X axis
      showlegend = FALSE,
      margin = list(t = 50) # Increase the top margin to move the title lower
    )

  #################
  ### solutions ###
  #################

  if(is.null(solutions)==FALSE){ # Si el usuario introduce algo en solutions...

    solutions<-unique(solutions) # Limpiamos repeticiones por si acaso.
    solutions.names<-character() # Guardaremos nombres aquí.

    # Fabricamos matriz con una línea tonta para ir metiendo soluciones pedidas.
    solutions.matrix <- matrix(NA,nrow=1,ncol=2)

    # Vamos metiendo las soluciones. Ordenar alfabéticamente.
    if ("corecenter" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,corecentervalue(v))
      solutions.names <- c(solutions.names,"core-center")
    }
    if ("nucleolus" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleolusvalue(v))
      solutions.names <- c(solutions.names,"nucleolus")
    }
    if ("nucleoluspc" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleoluspcvalue(v))
      solutions.names <- c(solutions.names,"nucleoluspc")
    }
    if ("shapleyvalue" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,shapleyvalue(v))
      solutions.names <- c(solutions.names,"Shapley value")
    }
    if ("tauvalue" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,tauvalue(v))
      solutions.names <- c(solutions.names,"tau value")
    }

    # Si la matriz tiene aún una sola línea es porque el usuario la lio.
    if (dim(solutions.matrix)[1]==1) {
      warning('None of the introduced solutions are valid.',call.=F)
    } else { # Si hay más de una línea es que al menos escribió una solución bien.

      solutions.color<-"limegreen"
      if (color=="limegreen") {
        solutions.color<-"cyan"
      }

      # Quitamos la línea inicial tonta:
      solutions.matrix <- matrix(c(t(solutions.matrix[-1,])),ncol=2,byrow=TRUE)
      if (dim(solutions.matrix)[1]!=length(solutions)) {
        warning('At least one invalid solution has been introduced.',call.=F)
      }

      # Ahora representamos las soluciones que tenemos almacenadas.
      fig <- fig %>%
        add_trace(
          x = solutions.matrix[,1],
          y = solutions.matrix[,2],
          type = "scatter",
          mode = "markers",
          marker = list(size = 10, color = solutions.color, symbol = "circle"),
          name = solutions.names
        ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    }

  }

  ###################
  ### allocations ###
  ###################

  if(is.null(allocations)==FALSE){ # Si el usuario introduce algo en allocations...

    allocations.color<-"red"
    if (color=="red") {
      allocations.color<-"darkorange"
    }

    if (is.null(dim(allocations))) { # Si introduce una sola allocation en forma de vector...
      allocations<-matrix(allocations,nrow=1,byrow=TRUE)
    }

    if (dim(allocations)[1]==1) { # Si se introduce una sola allocation...
      allocations.names<-"allocation"
      if (dim(allocations)[2]!=2) { # Si la allocation no tienen 2 coordenadas, avisamos.
        warning('The introduced allocation is not consistent with the number of players in v.',call.=F)
      }
      if (sum(allocations)!=v[3]) { # avisamos si no es eficiente.
        warning('The introduced allocation is not efficient.',call.=F)
      }
    } else { # Si se introduce más de una allocation...
      allocations.names<-character()
      for (i in 1:dim(allocations)[1]) {
        allocations.names<-c(allocations.names,paste("allocation",i))
        if (dim(allocations)[2]!=2) { # Si las allocations no tienen 3 coordenadas, avisamos.
          warning('The introduced allocations are not consistent with the number of players in v.',call.=F)
        }
        if (sum(allocations[i,])!=v[3]) { # avisamos si no son eficientes.
          warning('allocation ',i,' is not efficient.',call.=F)
        }
      }
    }

    fig <- fig %>%
      add_trace(
        x = allocations[,1],
        y = allocations[,2],
        type = "scatter",
        mode = "markers",
        marker = list(size = 6, color = allocations.color, symbol = "circle"),
        name = allocations.names
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
  }

  #############################
  ### imputaciones y núcleo ###
  #############################

  fig <- fig %>%
    add_trace(data = points, x = ~x, y = ~y, type = 'scatter', mode = 'lines+markers',
              marker = list(color = color, size = 6), line = list(color = color),
              name = "core set vertex")
  message("For two-player games, the core set and the set of imputations are equal.")

  print(fig); return(invisible(NULL))

}

################################################################################
### plot3proj ##################################################################
################################################################################

plot3proj <- function(v,imputations=TRUE,solutions=NULL,allocations=NULL,color="blue") {

  #####################################
  ### comprobaciones y definiciones ###
  #####################################

  if(length(v) != 7){
    stop("The length of 'v' is not 7, so 'v' is not the characteristic function of a three-player game.")
  }
  # v_str <- paste(deparse(substitute(v)),"=[", paste(v, collapse = ","), "]", sep = "")
  v_str <- paste("[", paste(v, collapse = ","), "]", sep = "")

  # Fabricamos la gráfica y la ajustamos
  fig <- plot_ly()
  fig <- fig %>% layout(
    title = list(text=paste("projected core of",v_str),font=list(color=color)),
    xaxis = list(title = "Player 1"),
    yaxis = list(title = "Player 2", scaleanchor = "x"), # Lock Y axis to X axis
    showlegend = FALSE,
    margin = list(t = 50) # Increase the top margin to move the title lower
  )

  #################
  ### solutions ###
  #################

  if(is.null(solutions)==FALSE){ # Si el usuario introduce algo en solutions...

    solutions<-unique(solutions) # Limpiamos repeticiones por si acaso.
    solutions.names<-character() # Guardaremos nombres aquí.

    # Fabricamos matriz con una línea tonta para ir metiendo soluciones pedidas.
    solutions.matrix <- matrix(NA,nrow=1,ncol=3)

    # Vamos metiendo las soluciones. Ordenar alfabéticamente.
    if ("corecenter" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,corecentervalue(v))
      solutions.names <- c(solutions.names,"core-center")
    }
    if ("nucleolus" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleolusvalue(v))
      solutions.names <- c(solutions.names,"nucleolus")
    }
    if ("nucleoluspc" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleoluspcvalue(v))
      solutions.names <- c(solutions.names,"nucleoluspc")
    }
    if ("shapleyvalue" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,shapleyvalue(v))
      solutions.names <- c(solutions.names,"Shapley value")
    }
    if ("tauvalue" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,tauvalue(v))
      solutions.names <- c(solutions.names,"tau value")
    }

    # Si la matriz tiene aún una sola línea es porque el usuario la lio.
    if (dim(solutions.matrix)[1]==1) {
      warning('None of the introduced solutions are valid.',call.=F)
    } else { # Si hay más de una línea es que al menos escribió una solución bien.

      solutions.color<-"limegreen"
      if (color=="limegreen") {
        solutions.color<-"cyan"
      }

      # Quitamos la línea inicial tonta:
      solutions.matrix <- matrix(c(t(solutions.matrix[-1,])),ncol=3,byrow=TRUE)
      if (dim(solutions.matrix)[1]!=length(solutions)) {
        warning('At least one invalid solution has been introduced.',call.=F)
      }

      # Ahora representamos las soluciones que tenemos almacenadas.
      fig <- fig %>%
        add_trace(
          x = solutions.matrix[,1],
          y = solutions.matrix[,2],
          type = "scatter",
          mode = "markers",
          marker = list(size = 6, color = solutions.color, symbol = "circle"),
          name = solutions.names
        ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    }

  }

  ###################
  ### allocations ###
  ###################

  if(is.null(allocations)==FALSE){ # Si el usuario introduce algo en allocations...

    allocations.color<-"red"
    if (color=="red") {
      allocations.color<-"darkorange"
    }

    if (is.null(dim(allocations))) { # Si introduce una sola allocation en forma de vector...
      allocations<-matrix(allocations,nrow=1,byrow=TRUE)
    }

    if (dim(allocations)[1]==1) { # Si se introduce una sola allocation...
      allocations.names<-"allocation"
      if (dim(allocations)[2]!=3) { # Si la allocation no tienen 3 coordenadas, avisamos.
        warning('The introduced allocation is not consistent with the number of players in v.',call.=F)
      }
      if (sum(allocations)!=v[7]) { # avisamos si no es eficiente.
        warning('The introduced allocation is not efficient.',call.=F)
      }
    } else { # Si se introduce más de una allocation...
      allocations.names<-character()
      for (i in 1:dim(allocations)[1]) {
        allocations.names<-c(allocations.names,paste("allocation",i))
        if (dim(allocations)[2]!=3) { # Si las allocations no tienen 3 coordenadas, avisamos.
          warning('The introduced allocations are not consistent with the number of players in v.',call.=F)
        }
        if (sum(allocations[i,])!=v[7]) { # avisamos si no son eficientes.
          warning('allocation ',i,' is not efficient.',call.=F)
        }
      }
    }

    fig <- fig %>%
      add_trace(
        x = allocations[,1],
        y = allocations[,2],
        type = "scatter",
        mode = "markers",
        marker = list(size = 6, color = allocations.color, symbol = "circle"),
        name = allocations.names
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
  }

  ################################
  ### conjunto de imputaciones ###
  ################################

  if(essentialcheck(v)==FALSE){
    message("The set of imputations of of 'v' is empty (and so is the core).")
    if(is.null(allocations)){
      return(NULL)
    }
    print(fig); return(invisible(NULL))
  }

  if(imputations==TRUE){ # Si el usuario nos pide representar imputaciones...

    v1 <- v[1]; v2 <- v[2]; v3 <- v[3]; v123 <- v[7]

    vertices <- data.frame(
      x = c(v1,v1,v123-v2-v3),
      y = c(v2,v123-v1-v3,v2),
      z = c(v123-v1-v2,v3,v3)
    )

    # Dibujar el conjunto de imputaciones usando plotly
    fig <- fig %>% add_trace(
      x = c(vertices$x,vertices$x[1]),
      y = c(vertices$y,vertices$y[1]),
      type = "scatter",
      mode = "lines",
      line = list(color = "black", width = 2),
      name="set of<br>imputations<br>vertex",showlegend=FALSE
    ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))

  }

  cv <- corevertices234(v)

  ##############
  ### núcleo ###
  ##############

  # Caso: 0 vértices
  if (is.null(cv)) {
    message(paste("The core of 'v' is empty."))
    print(fig); return(invisible(NULL))
  }

  # Caso: 1 vértice
  if(dim(cv)[1]==1){
    message(paste("The core of 'v' is made of a single point."))
    fig <- fig %>%
      add_trace(
        x = cv[,1],
        y = cv[,2],
        type = "scatter",
        mode = "markers",
        marker = list(size = 6, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig); return(invisible(NULL))
  }

  # Caso: 2 vértices
  if(dim(cv)[1]==2){
    message("The core of 'v' is the segment between two points.")
    fig <- fig %>%
      add_trace(
        x = cv[,1],
        y = cv[,2],
        type = "scatter",
        mode = "lines+markers",
        line = list(color = color, width = 2),
        marker = list(size = 6, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig); return(invisible(NULL))
  }

  # Caso: 3 vértices
  if(dim(cv)[1]==3){
    message("The core of 'v' is a triangle.")
    # OJO: hay que "repetir el primer punto" para cerrar el triángulo.
    fig <- fig %>%
      add_trace(
        x = c(cv[,1],cv[1,1]),
        y = c(cv[,2],cv[1,2]),
        type = "scatter",
        mode = "lines+markers",
        line = list(color = color, width = 2),
        marker = list(size = 6, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig); return(invisible(NULL))
  }

  # Caso: más de 3 vértices (contenidos en el plano)
  # Si llegamos hasta aquí es que el núcleo no tiene nada raro.
  message("The core of 'v' is a polygon with more than three edges.")
  # Como hay más de 3 vértices, tenemos que ordenarlos.
  cv.orde<-polygonal(cv)$CO
  # OJO: hay que "repetir el primer punto" para cerrar el polígono.
  fig <- fig %>%
    add_trace(
      x = c(cv.orde[,1],cv.orde[1,1]),
      y = c(cv.orde[,2],cv.orde[1,2]),
      type = "scatter",
      mode = "lines+markers",
      line = list(color = color, width = 2),
      marker = list(size = 6, color = color),
      name="core<br>set<br>vertex",showlegend=FALSE
    ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
  print(fig); return(invisible(NULL))

}

################################################################################
### plot3real ##################################################################
################################################################################

plot3real <- function(v,imputations=TRUE,solutions=NULL,allocations=NULL,color="blue") {

  #####################################
  ### comprobaciones y definiciones ###
  #####################################

  if(length(v) != 7){
    stop("The length of 'v' is not 7, so 'v' is not the characteristic function of a three-player game.")
  }
  # v_str <- paste(deparse(substitute(v)),"=[", paste(v, collapse = ","), "]", sep = "")
  v_str <- paste("[", paste(v, collapse = ","), "]", sep = "")

  # Fabricamos la gráfica y la ajustamos
  fig <- plot_ly()
  fig <- fig %>% layout(title = list(text=paste("core of",v_str),
                                     font=list(color=color)),
                        margin = list(t = 50),
                        showlegend = FALSE,
                        scene = list(
                          xaxis = list(title = "Player 1"),
                          yaxis = list(title = "Player 2"),
                          zaxis = list(title = "Player 3")
                        ))

  #################
  ### solutions ###
  #################

  if(is.null(solutions)==FALSE){ # Si el usuario introduce algo en solutions...

    solutions<-unique(solutions) # Limpiamos repeticiones por si acaso.
    solutions.names<-character() # Guardaremos nombres aquí.

    # Fabricamos matriz con una línea tonta para ir metiendo soluciones pedidas.
    solutions.matrix <- matrix(NA,nrow=1,ncol=3)

    # Vamos metiendo las soluciones. Ordenar alfabéticamente.
    if ("corecenter" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,corecentervalue(v))
      solutions.names <- c(solutions.names,"core-center")
    }
    if ("nucleolus" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleolusvalue(v))
      solutions.names <- c(solutions.names,"nucleolus")
    }
    if ("nucleoluspc" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleoluspcvalue(v))
      solutions.names <- c(solutions.names,"nucleoluspc")
    }
    if ("shapleyvalue" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,shapleyvalue(v))
      solutions.names <- c(solutions.names,"Shapley value")
    }
    if ("tauvalue" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,tauvalue(v))
      solutions.names <- c(solutions.names,"tau value")
    }

    # Si la matriz tiene aún una sola línea es porque el usuario la lio.
    if (dim(solutions.matrix)[1]==1) {
      warning('None of the introduced solutions are valid.',call.=F)
    } else { # Si hay más de una línea es que al menos escribió una solución bien.

      solutions.color<-"limegreen"
      if (color=="limegreen") {
        solutions.color<-"cyan"
      }

      # Quitamos la línea inicial tonta:
      solutions.matrix <- matrix(c(t(solutions.matrix[-1,])),ncol=3,byrow=TRUE)
      if (dim(solutions.matrix)[1]!=length(solutions)) {
        warning('At least one invalid solution has been introduced.',call.=F)
      }

      # Ahora representamos las soluciones que tenemos almacenadas.
      fig <- fig %>%
        add_trace(
          x = solutions.matrix[,1],
          y = solutions.matrix[,2],
          z = solutions.matrix[,3],
          type = "scatter3d",
          mode = "markers",
          marker = list(size = 4, color = solutions.color, symbol = "circle"),
          name = solutions.names
        ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    }

  }

  ###################
  ### allocations ###
  ###################

  if(is.null(allocations)==FALSE){ # Si el usuario introduce algo en allocations...

    allocations.color<-"red"
    if (color=="red") {
      allocations.color<-"darkorange"
    }

    if (is.null(dim(allocations))) { # Si introduce una sola allocation en forma de vector...
      allocations<-matrix(allocations,nrow=1,byrow=TRUE)
    }

    if (dim(allocations)[1]==1) { # Si se introduce una sola allocation...
      allocations.names<-"allocation"
      if (dim(allocations)[2]!=3) { # Si la allocation no tienen 3 coordenadas, avisamos.
        warning('The introduced allocation is not consistent with the number of players in v.',call.=F)
      }
      if (sum(allocations)!=v[7]) { # avisamos si no es eficiente.
        warning('The introduced allocation is not efficient.',call.=F)
      }
    } else { # Si se introduce más de una allocation...
      allocations.names<-character()
      for (i in 1:dim(allocations)[1]) {
        allocations.names<-c(allocations.names,paste("allocation",i))
        if (dim(allocations)[2]!=3) { # Si las allocations no tienen 3 coordenadas, avisamos.
          warning('The introduced allocations are not consistent with the number of players in v.',call.=F)
        }
        if (sum(allocations[i,])!=v[7]) { # avisamos si no son eficientes.
          warning('allocation ',i,' is not efficient.',call.=F)
        }
      }
    }

    fig <- fig %>%
      add_trace(
        x = allocations[,1],
        y = allocations[,2],
        z = allocations[,3],
        type = "scatter3d",
        mode = "markers",
        marker = list(size = 4, color = allocations.color, symbol = "circle"),
        name = allocations.names
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
  }

  ################################
  ### conjunto de imputaciones ###
  ################################

  if(essentialcheck(v)==FALSE){
    message("The set of imputations of 'v' is empty (and so is the core).")
    if(is.null(allocations)){
      return(NULL)
    }
    print(fig)
    return(invisible(NULL))
  }

  if(imputations==TRUE){ # Si el usuario nos pide representar imputaciones...

    v1 <- v[1]; v2 <- v[2]; v3 <- v[3]; v123 <- v[7]

    vertices <- data.frame(
      x = c(v1,v1,v123-v2-v3),
      y = c(v2,v123-v1-v3,v2),
      z = c(v123-v1-v2,v3,v3)
    )

    # Dibujar el conjunto de imputaciones usando plotly
    fig <- fig %>% add_trace(
      x = c(vertices$x,vertices$x[1]),
      y = c(vertices$y,vertices$y[1]),
      z = c(vertices$z,vertices$z[1]),
      type = "scatter3d",
      mode = "lines",
      line = list(color = "black", width = 4),
      name="set of<br>imputations<br>vertex",showlegend=FALSE
    ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))

  }

  cv <- corevertices234(v)

  ##############
  ### núcleo ###
  ##############

  # Caso: 0 vértices
  if (is.null(cv)) {
    message(paste("The core of 'v' is empty."))
    print(fig)
    return(invisible(NULL))
  }

  # Caso: 1 vértice
  if(dim(cv)[1]==1){
    message(paste("The core of 'v' is made of a single point."))
    fig <- fig %>%
      add_trace(
        x = cv[,1],
        y = cv[,2],
        z = cv[,3],
        type = "scatter3d",
        mode = "markers",
        marker = list(size = 3, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig)
    return(invisible(NULL))
  }

  # Caso: 2 vértices
  if(dim(cv)[1]==2){
    message(paste("The core of 'v' is the segment between two points."))
    fig <- fig %>%
      add_trace(
        x = cv[,1],
        y = cv[,2],
        z = cv[,3],
        type = "scatter3d",
        mode = "lines+markers",
        line = list(color = color, width = 3),
        marker = list(size = 3, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig)
    return(invisible(NULL))
  }

  # Caso: 3 vértices
  if(dim(cv)[1]==3){
    message(paste("The core of 'v' is a triangle."))
    # OJO: hay que "repetir el primer punto" para cerrar el triángulo.
    fig <- fig %>%
      add_trace(
        x = c(cv[,1],cv[1,1]),
        y = c(cv[,2],cv[1,2]),
        z = c(cv[,3],cv[1,3]),
        type = "scatter3d",
        mode = "lines+markers",
        line = list(color = color, width = 3),
        marker = list(size = 3, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig)
    return(invisible(NULL))
  }

  # Caso: más de 3 vértices (contenidos en el plano)
  # Si llegamos hasta aquí es que el núcleo no tiene nada raro.
  message(paste("The core of 'v' is a polygon with more than three edges."))
  # Como hay más de 3 vértices, tenemos que ordenarlos.
  cv.orde<-polygonal(cv)$CO
  # OJO: hay que "repetir el primer punto" para cerrar el polígono.
  fig <- fig %>%
    add_trace(
      x = c(cv.orde[,1],cv.orde[1,1]),
      y = c(cv.orde[,2],cv.orde[1,2]),
      z = c(cv.orde[,3],cv.orde[1,3]),
      type = "scatter3d",
      mode = "lines+markers",
      line = list(color = color, width = 3),
      marker = list(size = 3, color = color),
      name="core<br>set<br>vertex",showlegend=FALSE
    ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
  print(fig)
  return(invisible(NULL))

}

################################################################################
### plot4proj ##################################################################
################################################################################

plot4proj <- function(v,imputations=TRUE,solutions=NULL,allocations=NULL,color="blue") {

  #####################################
  ### comprobaciones y definiciones ###
  #####################################

  if(length(v) != 15){
    stop("The length of 'v' is not 15, so 'v' is not the characteristic function of a four-player game.")
  }
  # v_str <- paste(deparse(substitute(v)),"=[", paste(v, collapse = ","), "]", sep = "")
  v_str <- paste("[", paste(v, collapse = ","), "]", sep = "")

  # Fabricamos la gráfica y la ajustamos
  fig <- plot_ly()
  fig <- fig %>% layout(title = list(text=paste("projected core of",v_str),
                                     font=list(color=color)),
                        margin = list(t = 50),
                        showlegend = FALSE,
                        scene = list(
                          xaxis = list(title = "Player 1"),
                          yaxis = list(title = "Player 2"),
                          zaxis = list(title = "Player 3")
                        ))

  #################
  ### solutions ###
  #################

  if(is.null(solutions)==FALSE){ # Si el usuario introduce algo en solutions...

    solutions<-unique(solutions) # Limpiamos repeticiones por si acaso.
    solutions.names<-character() # Guardaremos nombres aquí.

    # Fabricamos matriz con una línea tonta para ir metiendo soluciones pedidas.
    solutions.matrix <- matrix(NA,nrow=1,ncol=4)

    # Vamos metiendo las soluciones. Ordenar alfabéticamente.
    if ("corecenter" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,corecentervalue(v))
      solutions.names <- c(solutions.names,"core-center")
    }
    if ("nucleolus" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleolusvalue(v))
      solutions.names <- c(solutions.names,"nucleolus")
    }
    if ("nucleoluspc" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleoluspcvalue(v))
      solutions.names <- c(solutions.names,"nucleoluspc")
    }
    if ("shapleyvalue" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,shapleyvalue(v))
      solutions.names <- c(solutions.names,"Shapley value")
    }
    if ("tauvalue" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,tauvalue(v))
      solutions.names <- c(solutions.names,"tau value")
    }

    # Si la matriz tiene aún una sola línea es porque el usuario la lio.
    if (dim(solutions.matrix)[1]==1) {
      warning('None of the introduced solutions are valid.',call.=F)
    } else { # Si hay más de una línea es que al menos escribió una solución bien.

      solutions.color<-"limegreen"
      if (color=="limegreen") {
        solutions.color<-"cyan"
      }

      # Quitamos la línea inicial tonta:
      solutions.matrix <- matrix(c(t(solutions.matrix[-1,])),ncol=4,byrow=TRUE)
      if (dim(solutions.matrix)[1]!=length(solutions)) {
        warning('At least one invalid solution has been introduced.',call.=F)
      }

      # Ahora representamos las soluciones que tenemos almacenadas.
      fig <- fig %>%
        add_trace(
          x = solutions.matrix[,1],
          y = solutions.matrix[,2],
          z = solutions.matrix[,3],
          type = "scatter3d",
          mode = "markers",
          marker = list(size = 4, color = solutions.color, symbol = "circle"),
          name = solutions.names
        ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    }

  }

  ###################
  ### allocations ###
  ###################

  if(is.null(allocations)==FALSE){ # Si el usuario introduce algo en allocations...

    allocations.color<-"red"
    if (color=="red") {
      allocations.color<-"darkorange"
    }

    if (is.null(dim(allocations))) { # Si introduce una sola allocation en forma de vector...
      allocations<-matrix(allocations,nrow=1,byrow=TRUE)
    }

    if (dim(allocations)[1]==1) { # Si se introduce una sola allocation...
      allocations.names<-"allocation"
      if (dim(allocations)[2]!=4) { # Si la allocation no tienen 4 coordenadas, avisamos.
        warning('The introduced allocation is not consistent with the number of players in v.',call.=F)
      }
      if (sum(allocations)!=v[15]) { # avisamos si no es eficiente.
        warning('The introduced allocation is not efficient.',call.=F)
      }
    } else { # Si se introduce más de una allocation...
      allocations.names<-character()
      for (i in 1:dim(allocations)[1]) {
        allocations.names<-c(allocations.names,paste("allocation",i))
        if (dim(allocations)[2]!=4) { # Si las allocations no tienen 4 coordenadas, avisamos.
          warning('The introduced allocations are not consistent with the number of players in v.',call.=F)
        }
        if (sum(allocations[i,])!=v[15]) { # avisamos si no son eficientes.
          warning('allocation ',i,' is not efficient.',call.=F)
        }
      }
    }

    fig <- fig %>%
      add_trace(
        x = allocations[,1],
        y = allocations[,2],
        z = allocations[,3],
        type = "scatter3d",
        mode = "markers",
        marker = list(size = 4, color = allocations.color, symbol = "circle"),
        name = allocations.names
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
  }

  ################################
  ### conjunto de imputaciones ###
  ### impu4proj ##################
  ################################

  if(essentialcheck(v)==FALSE){
    message("The set of imputations of 'v' is empty (and so is the core).")
    if(is.null(allocations)){
      return(NULL)
    }
    print(fig); return(invisible(NULL))
  }

  if(imputations==TRUE){ # Si el usuario nos pide representar imputaciones...

    v1 <- v[1]; v2 <- v[2]; v3 <- v[3]; v4 <- v[4]
    v1234 <- v[15]
    vertices <- matrix(c(
      v1, v2, v3,
      v1, v2, v1234 - v1 - v2 - v4,
      v1, v1234 - v1 - v3 - v4, v3,
      v1234 - v2 - v3 - v4, v2, v3
    ), ncol = 3, byrow = TRUE)

    faces <- matrix(c(1, 2, 3, 1, 3, 4, 1, 2, 4, 2, 3, 4), ncol = 3, byrow = TRUE)
    edges <- matrix(c(
      1, 2, 1, 3, 1, 4,
      2, 3, 2, 4, 3, 4
    ), ncol = 2, byrow = TRUE)

    # METER AQUÍ #

    for (i in 1:nrow(edges)) {
      fig <- fig %>% add_trace(
        type = "scatter3d", mode = "lines",
        x = vertices[edges[i, ], 1],
        y = vertices[edges[i, ], 2],
        z = vertices[edges[i, ], 3],
        line = list(color = "black", width = 4),
        name="set of<br>imputations<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    }

  }

  cv <- corevertices234(v)

  #########################
  ### casos degenerados ###
  #########################

  # Caso: 0 vértices
  if (is.null(cv)) {
    message(paste("The core of 'v' is empty."))
    print(fig); return(invisible(NULL))
  }

  # Caso: 1 vértice
  if(dim(cv)[1]==1){
    message(paste("The core of 'v' is made of a single point."))
    fig <- fig %>%
      add_trace(
        x = cv[,1],
        y = cv[,2],
        z = cv[,3],
        type = "scatter3d",
        mode = "markers",
        marker = list(size = 3, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig); return(invisible(NULL))
  }

  # Caso: 2 vértices
  if(dim(cv)[1]==2){
    message(paste("The core of 'v' is the segment between two points."))
    fig <- fig %>%
      add_trace(
        x = cv[,1],
        y = cv[,2],
        z = cv[,3],
        type = "scatter3d",
        mode = "lines+markers",
        line = list(color = color, width = 3),
        marker = list(size = 3, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig); return(invisible(NULL))
  }

  # Caso: 3 vértices
  if(dim(cv)[1]==3){
    message(paste("The core of 'v' is a triangle."))
    # OJO: hay que "repetir el primer punto" para cerrar el triángulo.
    fig <- fig %>%
      add_trace(
        x = c(cv[,1],cv[1,1]),
        y = c(cv[,2],cv[1,2]),
        z = c(cv[,3],cv[1,3]),
        type = "scatter3d",
        mode = "lines+markers",
        line = list(color = color, width = 3),
        marker = list(size = 3, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
    print(fig); return(invisible(NULL))
  }

  # Caso: más de 3 vértices contenidos en un plano
  if(dim(cv)[1]>3){
    # Vemos si los vértices son coplanarios.
    coplanarios<-TRUE # Coplanarios hasta que se demuestre lo contrario.
    cv.proj<-cv[,1:3] # Nos quedamos solo con las 3 primeras coordenadas.
    normal <- prodvect(cv.proj[2,]-cv.proj[1,],cv.proj[3,]-cv.proj[1,])
    for (i in 4:nrow(cv)) {
      vec <- cv.proj[i,]-cv.proj[1,]
      if (abs(sum(normal * vec)) > 1e-6) { # Si algún prod. escalar no es 0...
        coplanarios<-FALSE # ... los vértices no son coplanarios.
        break # No necesitamos ver más.
      }
    }
    if(coplanarios==TRUE){
      message(paste("The core of 'v' is a polygon with more than three edges."))
      # Como hay más de 3 vértices, tenemos que ordenarlos.
      cv.orde<-polygonal(cv.proj)$CO
      # OJO: hay que "repetir el primer punto" para cerrar el polígono.
      fig <- fig %>%
        add_trace(
          x = c(cv.orde[,1],cv.orde[1,1]),
          y = c(cv.orde[,2],cv.orde[1,2]),
          z = c(cv.orde[,3],cv.orde[1,3]),
          type = "scatter3d",
          mode = "lines+markers",
          line = list(color = color, width = 3),
          marker = list(size = 3, color = color),
          name="core<br>set<br>vertex",showlegend=FALSE
        ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
      print(fig); return(invisible(NULL))
    }
  }

  ############################
  ### casos no degenerados ###
  ### core4proj ##############
  ############################

  # Si llegamos hasta aquí es porque el núcleo tiene tres dimensiones.

  # Usamos la orden convhulln para tener los vectores de cada cara, con paquete geometry
  hull <- convhulln(cv[,1:3], return.non.triangulated.facets=TRUE)

  # Creamos una lista para almacenar los vectores de cada cara
  faces <- vector("list", nrow(hull))

  # Configuramos la lista con los vectores correspondientes
  for (i in 1:nrow(hull)) {
    indices <- na.omit(hull[i, ])  # Eliminar NAs
    faces[[i]] <- cv[indices, 1:3]  # Extraer las filas correspondientes, todas menos la última
    faces[[i]] <- as.matrix(faces[[i]]) # convertimos en matriz para llamar a polygonal
  }

  # Ordenamos los vectores de las caras
  ordeblue_faces <- lapply(faces, polygonal)

  # Iteraramos sobre cada cara en ordeblue_faces y la agregamos al gráfico
  for (i in seq_along(ordeblue_faces)) {
    # Cerrar el polígono agregando el primer punto al final
    sorted_points <- rbind(ordeblue_faces[[i]]$CO, ordeblue_faces[[i]]$CO[1,])

    # Agregar la traza a la figura
    fig <- fig %>%
      add_trace(
        x = sorted_points[,1],
        y = sorted_points[,2],
        z = sorted_points[,3],
        type = "scatter3d",
        mode = "lines+markers",
        line = list(color = color, width = 3),
        marker = list(size = 3, color = color),
        name="core<br>set<br>vertex",showlegend=FALSE
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
  }

  print(fig); return(invisible(NULL))
}

################################################################################
### plot4real ##################################################################
################################################################################

plot4real <- function(v,imputations=TRUE,solutions=NULL,allocations=NULL,color="blue") {

  #####################################
  ### comprobaciones y definiciones ###
  #####################################

  if(length(v) != 15){
    stop("The length of 'v' is not 15, so 'v' is not the characteristic function of a four-player game.")
  }

  # Asignar valores de v a variables individuales
  v1 <- v[1]; v2 <- v[2]; v3 <- v[3]; v4 <- v[4]
  v12 <- v[5]; v13 <- v[6]; v14 <- v[7]; v23 <- v[8]; v24 <- v[9]; v34 <- v[10]
  v123 <- v[11]; v124 <- v[12]; v134 <- v[13]; v234 <- v[14]
  v1234 <- v[15]

  # Longitud L del lado del tetraedro de imputaciones
  ll <- v1234 - v1 - v2 - v3 - v4

  if (ll < 0) {
    message("The set of imputatations of 'v' is empty (and so is the core).")
    return(NULL)
  }

  if (ll == 0) {
    if (belong2corecheck(v,x=c(v1,v2,v3,v4))) {
      message(sprintf("The core and the set of imputations of 'v' are made of a single point: (%f, %f, %f, %f)", v1, v2, v3, v4))
      return(NULL)
    } else {
      message(sprintf("The core of 'v' is empty and the set of imputations is made of a single point: (%f, %f, %f, %f)", v1, v2, v3, v4))
      return(NULL)
    }
  }

  # Fabricamos la gráfica y la ajustamos
  # v_str <- paste(deparse(substitute(v)),"=[", paste(v, collapse = ","), "]", sep = "")
  v_str <- paste("[", paste(v, collapse = ","), "]", sep = "")
  fig <- plot_ly()
  fig <- fig %>% layout(title = list(text=paste("'real' core of",v_str),
                                     font=list(color=color)),
                        margin = list(t = 50),
                        showlegend = FALSE,
                        scene = list(
                          xaxis = list(showticklabels = FALSE, title = " "),
                          yaxis = list(showticklabels = FALSE, title = " "),
                          zaxis = list(showticklabels = FALSE, title = " ")
                        ))

  #################
  ### solutions ###
  #################

  if(is.null(solutions)==FALSE){ # Si el usuario introduce algo en solutions...

    solutions<-unique(solutions) # Limpiamos repeticiones por si acaso.
    solutions.names<-character() # Guardaremos nombres aquí.

    # Fabricamos matriz con una línea tonta para ir metiendo soluciones pedidas.
    solutions.matrix <- matrix(NA,nrow=1,ncol=4)

    # Vamos metiendo las soluciones. Ordenar alfabéticamente.
    if ("corecenter" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,corecentervalue(v))
      solutions.names <- c(solutions.names,"core-center")
    }
    if ("nucleolus" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleolusvalue(v))
      solutions.names <- c(solutions.names,"nucleolus")
    }
    if ("nucleoluspc" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,nucleoluspcvalue(v))
      solutions.names <- c(solutions.names,"nucleoluspc")
    }
    if ("shapleyvalue" %in% solutions) {
      solutions.matrix <- rbind(solutions.matrix,shapleyvalue(v))
      solutions.names <- c(solutions.names,"Shapley value")
    }
    if ("tauvalue" %in% solutions) {
      solutions.matrix<-rbind(solutions.matrix,tauvalue(v))
      solutions.names <- c(solutions.names,"tau value")
    }

    # Si la matriz tiene aún una sola línea es porque el usuario la lio.
    if (dim(solutions.matrix)[1]==1) {
      warning('None of the introduced solutions are valid.',call.=F)
    } else { # Si hay más de una línea es que al menos escribió una solución bien.

      solutions.color<-"limegreen"
      if (color=="limegreen") {
        solutions.color<-"cyan"
      }

      # Quitamos la línea inicial tonta:
      solutions.matrix <- matrix(c(t(solutions.matrix[-1,])),ncol=4,byrow=TRUE)
      if (dim(solutions.matrix)[1]!=length(solutions)) {
        warning('At least one invalid solution has been introduced.',call.=F)
      }

      # Calculamos las coords nuevas de las solutions:
      solutions.matrix.new <- t(projectionmatrix4(v) %*% t(solutions.matrix))

      # Ahora representamos las soluciones que tenemos almacenadas.
      fig <- fig %>%
        add_trace(
          x = solutions.matrix.new[,1],
          y = solutions.matrix.new[,2],
          z = solutions.matrix.new[,3],
          type = "scatter3d",
          mode = "markers",
          marker = list(size = 4, color = solutions.color, symbol = "circle"),
          text = solutions.names,showlegend=FALSE,hoverinfo="text"
        ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))

    }

  }

  ###################
  ### allocations ###
  ###################

  if(is.null(allocations)==FALSE){ # Si el usuario introduce algo en allocations...

    allocations.color<-"red"
    if (color=="red") {
      allocations.color<-"darkorange"
    }

    if (is.null(dim(allocations))) { # Si introduce una sola allocation en forma de vector...
      allocations<-matrix(allocations,nrow=1,byrow=TRUE)
    }

    if (dim(allocations)[1]==1) { # Si se introduce una sola allocation...
      allocations.names<-"allocation"
      if (dim(allocations)[2]!=4) { # Si la allocation no tienen 4 coordenadas, avisamos.
        warning('The introduced allocation is not consistent with the number of players in v.',call.=F)
      }
      if (sum(allocations)!=v[15]) { # avisamos si no es eficiente.
        warning('The introduced allocation is not efficient.',call.=F)
      }
    } else { # Si se introduce más de una allocation...
      allocations.names<-character()
      for (i in 1:dim(allocations)[1]) {
        allocations.names<-c(allocations.names,paste("allocation",i))
        if (dim(allocations)[2]!=4) { # Si las allocations no tienen 4 coordenadas, avisamos.
          warning('The introduced allocations are not consistent with the number of players in v.',call.=F)
        }
        if (sum(allocations[i,])!=v[15]) { # avisamos si no son eficientes.
          warning('allocation ',i,' is not efficient.',call.=F)
        }
      }
    }

    # Calculamos las coords nuevas de las allocations:
    allocations.new <- t(projectionmatrix4(v) %*% t(allocations))

    fig <- fig %>%
      add_trace(
        x = allocations.new[,1],
        y = allocations.new[,2],
        z = allocations.new[,3],
        type = "scatter3d",
        mode = "markers",
        marker = list(size = 4, color = allocations.color, symbol = "circle"),
        text = allocations.names,showlegend=FALSE,hoverinfo="text"
      ) %>% layout(hoverlabel = list(namelength=-1,font=list(size = 11)))
  }

  ################################
  ### conjunto de imputaciones ###
  ### impu4real ##################
  ################################

  if(imputations==TRUE){

    # Construcción de la matriz de vértices de imputaciones en R3
    vertices <- data.frame(
      x = c(0, ll, ll/2, ll/2, 0, ll/2),
      y = c(0, 0, sqrt(3)/2 * ll, sqrt(3)/6 * ll, 0, sqrt(3)/2 * ll),
      z = c(0, 0, 0, sqrt(6)/3 * ll, 0, 0)
    )

    # Definir las conexiones entre los vértices para trazar las líneas
    edges <- data.frame(
      x = c(vertices$x[1], vertices$x[2], vertices$x[3], vertices$x[4], vertices$x[5], vertices$x[6], vertices$x[2], vertices$x[4]),
      y = c(vertices$y[1], vertices$y[2], vertices$y[3], vertices$y[4], vertices$y[5], vertices$y[6], vertices$y[2], vertices$y[4]),
      z = c(vertices$z[1], vertices$z[2], vertices$z[3], vertices$z[4], vertices$z[5], vertices$z[6], vertices$z[2], vertices$z[4])
    )

    # Dibujar el conjunto de imputaciones usando plotly
    fig <- fig %>% add_trace(
      type = "scatter3d", mode = "lines",
      x = edges$x, y = edges$y, z = edges$z,
      line = list(color = "black", width = 4),
      name=" ",showlegend=FALSE,hoverinfo="none"
    )

  }

  cv <- corevertices234(v)

  #########################
  ### casos degenerados ###
  #########################

  # Caso: 0 vértices
  if (is.null(cv)) {
    message(paste("The core of 'v' is empty."))
    print(fig); return(invisible(NULL))
  }

  # Calculamos las coords nuevas de los corevertices:
  cv.new <- t(projectionmatrix4(v) %*% t(cv))

  # Caso: 1 vértice
  if(dim(cv)[1]==1){
    message(paste("The core of 'v' is made of a single point."))
    fig <- fig %>%
      add_trace(
        x = cv.new[,1],
        y = cv.new[,2],
        z = cv.new[,3],
        type = "scatter3d",
        mode = "markers",
        marker = list(size = 3, color = color),
        name=" ",showlegend=FALSE,hoverinfo="none"
      )
    print(fig); return(invisible(NULL))
  }

  # Caso: 2 vértices
  if(dim(cv)[1]==2){
    message(paste("The core of 'v' is the segment between two points."))
    fig <- fig %>%
      add_trace(
        x = cv.new[,1],
        y = cv.new[,2],
        z = cv.new[,3],
        type = "scatter3d",
        mode = "lines+markers",
        line = list(color = color, width = 3),
        marker = list(size = 3, color = color),
        name=" ",showlegend=FALSE,hoverinfo="none"
      )
    print(fig); return(invisible(NULL))
  }

  # Caso: 3 vértices
  if(dim(cv)[1]==3){
    message(paste("The core of 'v' is a triangle."))
    # OJO: hay que "repetir el primer punto" para cerrar el triángulo.
    fig <- fig %>%
      add_trace(
        x = c(cv.new[,1],cv.new[1,1]),
        y = c(cv.new[,2],cv.new[1,2]),
        z = c(cv.new[,3],cv.new[1,3]),
        type = "scatter3d",
        mode = "lines+markers",
        line = list(color = color, width = 3),
        marker = list(size = 3, color = color),
        name=" ",showlegend=FALSE,hoverinfo="none"
      )
    print(fig); return(invisible(NULL))
  }

  # Caso: más de 3 vértices contenidos en un plano
  if(dim(cv)[1]>3){
    # Vemos si los vértices son coplanarios.
    coplanarios<-TRUE # Coplanarios hasta que se demuestre lo contrario.
    normal <- prodvect(cv.new[2,]-cv.new[1,],cv.new[3,]-cv.new[1,])
    for (i in 4:nrow(cv)) {
      vec <- cv.new[i,]-cv.new[1,]
      if (abs(sum(normal * vec)) > 1e-6) { # Si algún prod. escalar no es 0...
        coplanarios<-FALSE # ... los vértices no son coplanarios.
        break # No necesitamos ver más.
      }
    }
    if(coplanarios==TRUE){
      message(paste("The core of 'v' is a polygon with more than three edges."))
      # Como hay más de 3 vértices, tenemos que ordenarlos.
      cv.new.orde<-polygonal(cv.new)$CO
      # OJO: hay que "repetir el primer punto" para cerrar el polígono.
      fig <- fig %>%
        add_trace(
          x = c(cv.new.orde[,1],cv.new.orde[1,1]),
          y = c(cv.new.orde[,2],cv.new.orde[1,2]),
          z = c(cv.new.orde[,3],cv.new.orde[1,3]),
          type = "scatter3d",
          mode = "lines+markers",
          line = list(color = color, width = 3),
          marker = list(size = 3, color = color),
          name=" ",showlegend=FALSE,hoverinfo="none"
        )
      print(fig); return(invisible(NULL))
    }
  }

  ############################
  ### casos no degenerados ###
  ### core4real ##############
  ############################

  # Si llegamos hasta aquí es porque el núcleo tiene tres dimensiones.

  # Usamos la orden convhulln para tener los vectores de cada cara, con paquete geometry
  hull <- convhulln(cv.new[,1:3], return.non.triangulated.facets=TRUE)

  # Creamos una lista para almacenar los vectores de cada cara
  faces <- vector("list", nrow(hull))

  # Configuramos la lista con los vectores correspondientes
  for (i in 1:nrow(hull)) {
    indices <- na.omit(hull[i, ])  # Eliminar NAs
    faces[[i]] <- cv.new[indices, 1:3]  # Extraer las filas correspondientes, todas menos la última
    faces[[i]] <- as.matrix(faces[[i]]) # convertimos en matriz para llamar a polygonal
  }

  # Ordenamos los vectores de las caras
  ordeblue_faces <- lapply(faces, polygonal)

  # Iteraramos sobre cada cara en ordeblue_faces y la agregamos al gráfico
  for (i in seq_along(ordeblue_faces)) {
    # Cerrar el polígono agregando el primer punto al final
    sorted_points <- rbind(ordeblue_faces[[i]]$CO, ordeblue_faces[[i]]$CO[1,])

    # Agregar la traza a la figura
    fig <- fig %>%
      add_trace(
        x = sorted_points[,1],
        y = sorted_points[,2],
        z = sorted_points[,3],
        type = "scatter3d",
        mode = "lines+markers",
        line = list(color = color, width = 3),
        marker = list(size = 3, color = color),
        name=" ",showlegend=FALSE,hoverinfo="none"
      )
  }

  print(fig); return(invisible(NULL))
}
