#' @title Lexicographic order to binary order
#' @description Given a characteristic function in lexicographic order, this function returns the characteristic function in binary order.
#' @param v A characteristic function, as a vector in lexicographic order.
#' @return The characteristic function, as a vector in binary order.
#' @details Lexicographic order arranges coalitions in ascending order according to size, and applies lexicographic order to break ties among coalitions of the same size. The binary order position of a coalition \eqn{S\in 2^N} is given by \eqn{\sum_{i\in S} 2^{i-1}}.
#' @examples
#' v <- seq(1:31)
#' lex2bin(v)
#' bin2lex(lex2bin(v))==v
#' @seealso \link{bin2lex}, \link{codebin2lex}, \link{codelex2bin}
#' @export

lex2bin <- function(v) {

  ################################
  ### Comprobación datos entrada###
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (n <= 2) # Si el juego es de dos jugadores, ambos juegos coinciden.
    {
      return(v)
    }

  ################################
  ##### Creación de variables######
  ################################

  vnew <- v # v en orden lexicográfico.

  ################################
  ###### Cuerpo de la función######
  ################################

  # Las dos primeras coaliciones y las tres últimas coinciden.
  for (jj in 3:(nC - 3))
  {
    # Paso de código lexicográfico a binario.
    vnew[codelex2bin(n, jj)] <- v[jj]
  }

  ################################
  ###### Salidas de la función#####
  ################################

  return(vnew)
} # Fin de la función.
