#' Extract depth layer information from root scans
#'
#' \code{getDepthLayerInfo} - Specify a set of root scan directories in the
#' form of a data frame (e.g., by using \code{getOverviewInput()}). Then, the
#' function will do the following: \cr
#' - If there are several depth windows of the same root scan (project, tube,
#' date and session have to be identical), then these will be stitched and
#' saved in directory of the scan of depth window 1. \cr
#' - Create depth layer masks corresponding to the specified
#' \code{depth_levels_cm}. \cr
#' - Computes a range of values from the scans (per depth layer: "pixels_root",
#' "pixels_bg", "skel_pixels_total", "skel_pixels_low3", "skel_pixels_3-7",
#' "skel_pixels_larg7","kimura_length6") and if (\code{save_csvs} = TRUE) saves
#' them as a csv file in the directory of the scan of depth window 1. \cr
#' - Returns a data frame containing values listed above of all scans.
#'
#' @param root_df Data frame (default = NULL) specifying the structure
#' of the root scan data. It can be created using the function
#' \code{getOverviewInput()}. The data frame must contain several mandatory
#' features and can comprise other optional features, some of which have a
#' functional importance and others have no influence but will be included
#' in the output data frame). Any feature not listed here is treated as
#' completely optional: \cr
#' - --------------- Mandatory --------------- \cr
#' - \code{dir_name_full}: Full path to directory.  \cr
#' - \code{dir_name}: name of directory.  \cr
#' - \code{depth_window}: depth-level/window. 1 indicates the scan highest scan,
#' 2 the next scan somewhat lower in ground, and so forth. \cr
#' - -------- Optional, but functionally important -------- \cr
#' - \code{top_side}: One of "left","top","right","bottom". Indicates where
#' the upper side of the image is. If not specified, "left" is used. \cr
#' - \code{tube_angle}: Installation angle of the tube (angle between surface
#' plane and tube above ground, in degrees, >0 and <90). If not specified, an
#' angle of 45 degrees is assumed. \cr
#' - \code{depth_highpoint_cm}: Depth in cm of the highest point in the
#' image, e.g., 0 indicating that the top border of the scan scratched the
#' surface of the soil or -5 indicating that it lies 5 cm deep in the soil.
#' This is mostly important for all scans of \code{depth_window} 1 (or the
#' lowest number if the depth windows are not 1,2,3,4,...) if all
#' images are stitched together. If not specified, it is assumed that the
#' top border of scan of \code{depth_window} 1 lies at depth 0. \cr
#' - \code{pos_highpoint_px}: Either one of "center" or "edge",
#' indicating that scanning starts at the bottom or top facing side of the
#' minirhizotron, respectively, or it can also be an integer value (>=1 and
#' <=width of the \code{top_side} of the image) specifying where (left<->right)
#' in the top row of the image the highest point of the image lies, indicating
#' the column where the minimum points of the depth-level lines lie. \cr
#' If not specified, "center" is assumed as scanning often starts at the
#' bottom facing side of the minirzhizotron. \cr
#' See also \code{gap_cm} if there is a non-zero gap. \cr
#' - \code{ppcm} Pixels per centimeter, the resolution of the image.
#' If not specified (NULL or NA), \code{ppi} is used.
#' Common resolutions are 300 ppi (or dpi, for print jobs) which is 118 px/cm,
#' 150 ppi which is 59 px/cm, and 72 ppi (for screens) which is 28 px/cm.\cr
#' - \code{ppi} Pixels per inch, the resolution of the image.
#' If \code{ppcm} and \code{ppi} arenot specified, 300 ppi is used.
#' Common resolutions are 300 ppi (or dpi, for print jobs) which is 118 px/cm,
#' 150 ppi which is 59 px/cm, and 72 ppi (for screens) which is 28 px/cm.\cr
#' - \code{overlap_px} Overlap in pixels of the current image and the image of
#' the previous depth window for the stitching process, i.e., this is not
#' important for the image of the first depth window.
#' While stitching it will be checked if there is a better overlap in a small
#' area around the specified value (see \code{max_shift_px}).
#' If not specified it is set to 1's.\cr
#' - \code{max_shift_px} Radius around \code{overlap_px} that is checked for
#' better overlap matching. If not specified it is set to 0 (no alternatives
#' are checked).\cr
#' - \code{gap_cm} Numeric value that specifies if there is a gap
#' (in cm) between the two "matching" sides of the image, i.e., there is no full
#' rotation of the scanner. If \code{pos_highpoint_px} is specified as "center"
#' or "edge" (in characters), then it is assumed that the middle of the gap is
#' right at the top or bottom facing side of the tube, i.e. it is not possible
#' to start and end at the exact top/bottom. If \code{pos_highpoint_px} is
#' specified as a numeric value, then it is considered as exact and the gap
#' does not move it.\cr
#' If not specified and if \code{gap_deg} is NA as well, no gap (0) is
#' assumed. \cr
#' - \code{gap_deg} Numeric value that specifies if there is a gap
#' (in degrees, i.e. from 0 to 360) between the two "matching" sides of the
#' image, i.e., there is no full rotation of the scanner.
#' Works similar to \code{gap_cm}.
#' Only used if \code{gap_cm} is not specified.\cr
#' - \code{project}: Project name. Needed to identify scans that should be
#' stitched. \cr
#' - \code{tube}: ID of the minirhizotron. Needed to identify scans that should
#' be stitched.  \cr
#' - \code{date}: Date of the scanning. Needed to identify scans that should be
#' stitched. \cr
#' - \code{session}: ID of the scan session. Needed to identify scans that
#' should be stitched.  \cr
#' - ------------ Completely optional ------------ \cr
#' - \code{ID}: ID of the scan (6 digits) or timecode. \cr
#' - \code{operator}: ID of the person that scanned the root. \cr
#' - \code{file_extension}: File format: png, tiff, jpg, or jpeg
#' (upper or lowercase).
#' @param depth_levels_cm Numeric matrix with two columns and at least
#' one row. Each row specifies a depth interval (in cm) that is of
#' interest. For each interval a masking layer is generated. The first value
#' of each row has to be larger than the second (decending).\cr
#' By default: Depth layers 0-(-10)cm, -10-(-20)cm, -20-(-30)cm, and
#' -30-(-40)cm.
#' @param overwrite If FALSE (default), root scan directories are skipped that
#' have already been processed by a particular procedure, i.e., there already
#' are stitched images before beginning the stitching procedure, there already
#' are images with depth layers drawn in before saving thme or there already is
#' a statistics_withDepthLayers.csv file when it comes to evaluating the image.
#' If TRUE, all procedures are applied to the scan and corresponding files are
#' overwritten.
#' @param save_csvs Specifies if csv files for the individual scans should be
#' saved in the scan's directory (default TRUE).
#' @param save_pngs_col If not NULL, the root scans with depth layers drawn in
#' are saved in the scan's directory. Specifies the color of the FALSE-regions
#' of the mask (default "grey"). Can be a vector of the same length as
#' the number of depth intervals to allow each layer to be drawn in with a
#' different color. \cr
#' Can be of any of the three kinds of R color specifications, i.e., either a
#' color name (as listed by colors()), a hexadecimal string (see Details), or a
#' positive integer (indicates using palette()).
#'
#' @return \code{getDepthLayerInfo} A data frame containing the depth layer
#' information of all individual scans.
#'
#' @export
#' @rdname getDepthLayerInfo
#'
#' @examples
#' getDepthLayerInfo()
getDepthLayerInfo <- function(root_df = NULL,
                              depth_levels_cm = rbind(c(0,-10),
                                                      c(-10,-20),
                                                      c(-20,-30),
                                                      c(-30,-40)),
                              overwrite = FALSE,
                              save_csvs = TRUE, save_pngs_col = "grey"){
  if(is.null(root_df)){
    message("No root data frame specified.\n")
    return(NULL)
  }
  message(paste("Getting an overview of the scans and checking",
                "if stitching is necessary."))
  # List all root image directories --------------------------------------------
  img_dirs <- standardizePaths(root_df$dir_name)
  img_dirs_full <- standardizePaths(root_df$dir_name_full)
  # 1: Sort and group the root image directories -------------------------------
  img_dirs_checked <- rep(FALSE, length(img_dirs))
  main_img_howmany <- rep(NA, length(img_dirs))
  other_lvls <- vector("list", length(img_dirs))
  # Check whether stitched images already exist.
  dir_contain_stitched <- sapply(img_dirs_full,
                           FUN = function(X){ return(
                             file.exists(paste0(X,"/",basename(X),
                                                ".stitched.segmentation.png")) &
                             file.exists(paste0(X,"/",basename(X),
                                                ".stitched.skeleton.png")) )})
  for(i in 1:length(img_dirs)){
    if(!img_dirs_checked[i]){
      # Find all directories that contain the different levels of the current
      # root scan.
      dirs_with_lvls <- which(root_df$project == root_df$project[i] &
                              root_df$tube == root_df$tube[i] &
                              root_df$date == root_df$date[i] &
                              root_df$session == root_df$session[i])
      # Which one has the smallest depth window number? -> is main scan
      i_min <- dirs_with_lvls[which.min(root_df$depth_window[dirs_with_lvls])]
      # Check if they are consecutive numbers.
      if(sum(sort(root_df$depth_window[dirs_with_lvls]) !=
             root_df$depth_window[i_min]+0:(length(dirs_with_lvls)-1))>0){
        stop(paste0("The depth windows are not consecutively numbered, i.e., ",
                    "1,2,3,4,... (see directory ", i,
                    " and cooresponding scans)."))
      }
      if(length(dirs_with_lvls)>1){
        # Keep track of all checked directories:
        other_lvls[[i_min]] <- as.vector(setdiff(sort(dirs_with_lvls,
                                                      decreasing = FALSE),
                                                 i_min))
      }
      main_img_howmany[i_min] <- length(dirs_with_lvls)
      img_dirs_checked[dirs_with_lvls] <- TRUE
    }
  }

  # 2: Stitch images of different depth-levels together ------------------------
  main_to_stitch <- which(main_img_howmany>1)
  if(!overwrite){
    main_to_stitch <- main_to_stitch[!dir_contain_stitched[main_to_stitch]]
    if(length(main_to_stitch)<length(which(main_img_howmany>1))){
      message("Stitching (partially) skipped.")
    }
  }
  if(length(main_to_stitch)==0){
    message("No stitching necessary.")
  } else {
    message(paste("Stitching necessary for the following directories:",
                  paste0(sapply(main_to_stitch,
                                function(X){
                                  return(paste0("(",X,", ",
                                                paste0(other_lvls[[X]],
                                                        collapse = ", "),
                                                ")"))}),
                                         collapse = ", ")))
    # Do the stitching for all scans of each group.-----------------------------
    progress_stitching = utils::txtProgressBar(min = 0,
                                               max = 2*length(main_to_stitch),
                                               initial = 0, style = 3)
    progress_stitching_counter <- 0
    for(curr_main in main_to_stitch){
      # 2a: Get some parameters from the data frame. ---------------------------
      # Current overlaps.
      curr_overlap_px <- root_df[other_lvls[[curr_main]],"overlap_px"]
      if(sum(is.null(curr_overlap_px)>0) ||
         sum(is.na(curr_overlap_px)>0)){
        curr_overlap_px <- NULL
      }
      # Current maximal shifts.
      curr_max_shift_px <- root_df[other_lvls[[curr_main]],"max_shift_px"]
      if(sum(is.null(curr_max_shift_px)>0) ||
         sum(is.na(curr_max_shift_px)>0)){
           curr_max_shift_px <- NULL
      }
      # Current stitch direction
      curr_top_side <- unique(root_df[other_lvls[[curr_main]],"top_side"])
      if(is.null(curr_top_side) || length(curr_top_side)>1 ||
         !curr_top_side %in% c(NA, "top", "left", "right", "bottom")){
        warning(paste0("There are contradicting inputs for 'top_side' in the ",
                       "data frame for rows ",
                       paste(c(curr_main,other_lvls[[curr_main]]),
                             collapse = ","),".\n Continuing with ",
                       "'left_to_right' stitch direction."))
        curr_stitch_direction <- "left_to_right"
      } else {
        if(is.na(curr_top_side)){
          curr_stitch_direction <- "left_to_right"
        } else if(curr_top_side== "top"){
          curr_stitch_direction <- "top_to_bottom"
        } else if(curr_top_side== "left"){
          curr_stitch_direction <- "left_to_right"
        } else if(curr_top_side== "right"){
          curr_stitch_direction <- "right_to_left"
        } else if(curr_top_side== "top"){
          curr_stitch_direction <- "bottom_to_top"
        }
      }
      # 2b: Stitch the segmentation images. ------------------------------------
      stitched_seg <- stitchImgs(
        img_paths = sapply(c(curr_main,other_lvls[[curr_main]]), function(X){
            paste0(img_dirs_full[X],"/",img_dirs[X],".segmentation.png")}),
        imgs = NULL,
        out_file = paste0(img_dirs_full[curr_main],"/",img_dirs[curr_main],
                          ".stitched.segmentation.png"),
        overlap_px = curr_overlap_px, max_shift_px = curr_max_shift_px,
        perc_better_per_px = 0.001, corr_formula = "weighted_matches_b_w",
        stitch_direction = curr_stitch_direction, blending_mode = "under",
        show_messages = TRUE)
      #showImgMask(stitched_seg)
      #attributes(stitched_seg)
      progress_stitching_counter <- progress_stitching_counter +1
      utils::setTxtProgressBar(progress_stitching,
                               progress_stitching_counter)
      # 2c: Stitch the skeleton images with the same overlaps. -----------------
      stitched_skel <- stitchImgs(
        img_paths = sapply(c(curr_main,other_lvls[[curr_main]]), function(X){
          paste0(img_dirs_full[X],"/",img_dirs[X],".skeleton.png")}),
        imgs = NULL,
        out_file = paste0(img_dirs_full[curr_main],"/", img_dirs[curr_main],
                          ".stitched.skeleton.png"),
        overlap_px = attr(stitched_seg, "overlap"), max_shift_px = 0,
        stitch_direction = curr_stitch_direction, blending_mode = "under",
        show_messages = TRUE)
      progress_stitching_counter <- progress_stitching_counter +1
      utils::setTxtProgressBar(progress_stitching,
                               progress_stitching_counter)
    }
    close(progress_stitching)
    message("Stitching finished.")
  }


  # 3: Create a data frame and extract the depth layer data --------------------
  message("Preparing data frame.")
  numb_scans <- sum(!is.na(main_img_howmany))
  main_scans <- which(!is.na(main_img_howmany))
  root_stats_wLvls <- root_df[,-which(names(root_df)=="dir_name_full")]
  root_stats_wLvls[,"stitched"] <- main_img_howmany
  # Initialize data frame with new columns for the different soil levels.
  for(lvl_index in 1:nrow(depth_levels_cm)){
    root_stats_wLvls[, paste0(c("pixels_root",
                                "skel_pixels_total", "skel_pixels_low3",
                                "skel_pixels_3-7", "skel_pixels_larg7",
                                "kimura_length6"),"_",
                              depth_levels_cm[lvl_index,1],"_",
                              depth_levels_cm[lvl_index,2])] <- rep(NA,
                                                                 nrow(root_df))
  }

  # 4: Fill the data frame by going through the main scans ---------------------
  message(paste("Starting to extract information of",
                numb_scans, "(stitched) scans."))
  progress_depthlvlinfo = utils::txtProgressBar(min = 0,
                                   max = (2+nrow(depth_levels_cm)) * numb_scans,
                                   initial = 0, style = 3)
  progress_depthlvlinfo_counter <- 0
  for(i in main_scans){
    if(!overwrite &&
       file.exists(paste0(img_dirs_full[i],"/statistics_withDepthLayers.csv"))){
      root_stats_wLvls[i,] <-  utils::read.csv(paste0(img_dirs_full[i],
                                      "/statistics_withDepthLayers.csv"))
      progress_depthlvlinfo_counter <- progress_depthlvlinfo_counter +
        (1+nrow(depth_levels_cm))
      utils::setTxtProgressBar(progress_depthlvlinfo,
                               progress_depthlvlinfo_counter)
    } else {
      # Read in the corresponding PNG files. If a stitched version exists,
      # use it.
      if(main_img_howmany[i]>1){
        if(file.exists(paste0(img_dirs_full[i], "/", img_dirs[i],
                              ".stitched.segmentation.png"))&&
           file.exists(paste0(img_dirs_full[i], "/", img_dirs[i],
                              ".stitched.skeleton.png"))){
          png_base <- png::readPNG(paste0(img_dirs_full[i], "/", img_dirs[i],
                                          ".stitched.segmentation.png"))
          png_skel <- png::readPNG(paste0(img_dirs_full[i], "/", img_dirs[i],
                                          ".stitched.skeleton.png"))
        } else {
          stop(paste("The stitched files do not exist. Directory: \n",
                     img_dirs_full[i]))
        }
      } else {
        png_base <- png::readPNG(paste0(img_dirs_full[i], "/", img_dirs[i],
                                        ".segmentation.png"))
        png_skel <- png::readPNG(paste0(img_dirs_full[i], "/", img_dirs[i],
                                        ".skeleton.png"))
      }
      # Check if there are unexpected colors.
      if(sum(rowSums(png_base, dims = 2) != 0 & # black (3x0)
             rowSums(png_base, dims = 2) != 1 & # red (1X1,2x0) (,green or blue)
             rowSums(png_base, dims = 2) != 3)>0 ){ # white(3x1)
        warning(paste("There are unexpected colors in the base png",
                      "image for",
                      img_dirs_full[i],
                      "Continuing the process by rounding the values."))
        png_base <- round(png_base)
      }
      if(sum(rowSums(png_skel, dims = 2) != 0 &  # black (3x0)
             rowSums(png_skel, dims = 2) != 1 & # red (1X1,2x0) (,green or blue)
             rowSums(png_skel, dims = 2) != 3)>0){ # white(3x1)
        warning(paste("There are unexpected colors in the skeleton",
                      "png image",
                      "for", img_dirs_full[i],
                      "Continuing the process by rounding the values."))
        png_skel <- round(png_skel)
      }
      # Check if both images have the same dimensions and if not, fill the
      # missing parts with black. ----------------------------------------------
      if(sum(dim(png_base)[1:2] == dim(png_skel)[1:2]) < 2 ){
        warning(paste0("The base and skeleton png have different dimensions. ",
                      "Continuing the process by filling gaps with black. ",
                      "Pixel difference: base - skeleton height ",
                      dim(png_base)[1]-dim(png_skel)[1],
                      "px (",round(100*(dim(png_base)[1]-dim(png_skel)[1])/
                                   dim(png_base)[1],2),"%), width ",
                      dim(png_base)[2]-dim(png_skel)[2],
                      "px (",round(100*(dim(png_base)[2]-dim(png_skel)[2])/
                                   dim(png_base)[2],2),"%)\n",
                      "Corresponding directory:", img_dirs_full[i]))
        new_dim <- c(max(dim(png_base)[1],dim(png_skel)[1]),
                     max(dim(png_base)[2],dim(png_skel)[2]),3)
        temp_png <- array(0, dim = new_dim)
        temp_png[1:dim(png_base)[1],
                 1:dim(png_base)[2],] <- png_base
        png_base <- temp_png
        temp_png <- array(0, dim = new_dim)
        temp_png[1:dim(png_skel)[1],
                 1:dim(png_skel)[2],] <- png_skel
        png_skel <- temp_png
      }

      # Extract all important information from the root_df data frame. ---------
      # Current top side (rotation of the image).
      if(is.na(root_df[i,"top_side"])){
        curr_top_side <- "left"
        root_stats_wLvls[i,"top_side"] <- "left"
      } else {
        curr_top_side <- root_df[i,"top_side"]
      }

      # Current resolution.
      if(is.null(root_df[i,"ppcm"]) || is.na(root_df[i,"ppcm"])){
        if(is.null(root_df[i,"ppi"]) || is.na(root_df[i,"ppi"])){
          curr_ppcm <- ppi2ppcm(300)
          root_stats_wLvls[i,"ppi"] <- 300
          root_stats_wLvls[i,"ppcm"] <- ppi2ppcm(300)
        } else {
          curr_ppcm <- ppi2ppcm(root_df[i,"ppi"])
          root_stats_wLvls[i,"ppcm"] <- curr_ppcm
        }
      } else {
        curr_ppcm <- root_df[i,"ppcm"]
      }

      # Current depth of the highpoint.
      if(is.na(root_df[i,"depth_highpoint_cm"])){
        curr_depth_highpoint_cm <- 0
        root_stats_wLvls[i,"depth_highpoint_cm"] <- 0
      } else {
        curr_depth_highpoint_cm <- root_df[i,"depth_highpoint_cm"]
      }

      # Current position of the hiphpoint.
      if(is.na(root_df[i,"pos_highpoint_px"])){
        curr_pos_highpoint_px <- "center"
        root_stats_wLvls[i,"pos_highpoint_px"] <- "center"
      } else {
        curr_pos_highpoint_px <- root_df[i,"pos_highpoint_px"]
        if(!curr_pos_highpoint_px %in% c("center", "edge")){
          curr_pos_highpoint_px <- as.integer(curr_pos_highpoint_px)
        }
      }

      # Current installation angle.
      if(is.na(root_df[i,"tube_angle"])){
        curr_tube_angle <- 45
        root_stats_wLvls[i,"tube_angle"] <- 45
      } else {
        curr_tube_angle <- root_df[i,"tube_angle"]
      }

      # Current gap size.
      if(is.na(root_df[i,"gap_cm"]) && is.na(root_df[i,"gap_deg"])){
        curr_gap_cm <- 0
        curr_gap_deg <- 0
      } else {
        curr_gap_cm <- root_df[i,"gap_cm"]
        curr_gap_deg <- root_df[i,"gap_deg"]
      }
      # Compute the masking layers to subset the root pictures. ----------------
      stack_masks <- createDepthLayerMasks(ppcm = curr_ppcm,
                                  dims_px = dim(png_base)[1:2],
                                  depth_levels_cm = depth_levels_cm,
                                  depth_highpoint_cm = curr_depth_highpoint_cm,
                                  pos_highpoint_px = curr_pos_highpoint_px,
                                  angle = curr_tube_angle,
                                  top_side = curr_top_side,
                                  gap_cm = curr_gap_cm,
                                  gap_deg = curr_gap_deg)


      progress_depthlvlinfo_counter <- progress_depthlvlinfo_counter + 1
      utils::setTxtProgressBar(progress_depthlvlinfo,
                               progress_depthlvlinfo_counter)
      # Fill in the corresponding columns of the data frame: -------------------
      for(lvl_index in 1:nrow(depth_levels_cm)){
        root_stats_wLvls[i, paste0("pixels_root_",
                                   depth_levels_cm[lvl_index,1],"_",
                                   depth_levels_cm[lvl_index,2])] <-
          kimuraLength(skel_img = png_base, formula = 97,
                       mask = stack_masks[[lvl_index]], strict_mask = TRUE,
                       show_messages = FALSE)
        root_stats_wLvls[i, paste0("skel_pixels_total_",
                                   depth_levels_cm[lvl_index,1],"_",
                                   depth_levels_cm[lvl_index,2])] <-
          kimuraLength(skel_img = png_skel, formula = 97,
                       mask = stack_masks[[lvl_index]], strict_mask = TRUE,
                       show_messages = FALSE)
        root_stats_wLvls[i, paste0(c("skel_pixels_low3_","skel_pixels_3-7_",
                                     "skel_pixels_larg7_"),
                                   depth_levels_cm[lvl_index,1],"_",
                                   depth_levels_cm[lvl_index,2])] <-
          skelPxWidth(png_base, png_skel, mask = stack_masks[[lvl_index]])
        root_stats_wLvls[i, paste0("kimura_length6_",
                                   depth_levels_cm[lvl_index,1],"_",
                                   depth_levels_cm[lvl_index,2])] <-
          kimuraLength(skel_img = png_skel, formula = 6,
                       mask = stack_masks[[lvl_index]], strict_mask = TRUE,
                       show_messages = FALSE)

        progress_depthlvlinfo_counter <- progress_depthlvlinfo_counter + 1
        utils::setTxtProgressBar(progress_depthlvlinfo,
                                 progress_depthlvlinfo_counter)
      }
    }

    if(save_csvs && (overwrite ||
       !file.exists(paste0(img_dirs_full[i],
                           "/statistics_withDepthLayers.csv")))){
      utils::write.csv(root_stats_wLvls[i,],
                file = paste0(img_dirs_full[i],
                              "/statistics_withDepthLayers.csv"),
                row.names = FALSE)
    }

    # If desired, save the images with their masks.
    if(!is.null(save_pngs_col)){
      if(length(save_pngs_col)==1){
        save_pngs_col <- rep(save_pngs_col, nrow(depth_levels_cm))
      }
      for(lvl_index in 1:nrow(depth_levels_cm)){
        seg_w_DL <- paste0(img_dirs_full[i],
                           "/base_masked_depthlvl_",
                           depth_levels_cm[lvl_index,1],"_",
                           depth_levels_cm[lvl_index,2],"cm.png")
        if(overwrite || !file.exists(seg_w_DL)){
             png::writePNG(applyImgMask(png_base,
                                        mask = stack_masks[[lvl_index]],
                                        mask_col = save_pngs_col[lvl_index]),
                           target = seg_w_DL)
        }
        skel_w_DL <- paste0(img_dirs_full[i],
                            "/skel_masked_depthlvl_",
                            depth_levels_cm[lvl_index,1],"_",
                            depth_levels_cm[lvl_index,2],"cm.png")
        if(overwrite || !file.exists(skel_w_DL)){
          png::writePNG(applyImgMask(png_skel, mask = stack_masks[[lvl_index]],
                                     mask_col = save_pngs_col[lvl_index]),
                        target = skel_w_DL)
        }
      }
    }
    progress_depthlvlinfo_counter <- progress_depthlvlinfo_counter + 1
    utils::setTxtProgressBar(progress_depthlvlinfo,
                             progress_depthlvlinfo_counter)
  }
  close(progress_depthlvlinfo)
  return(root_stats_wLvls[main_scans,])
}
