% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/paths.R, R/print.paths.R
\name{paths}
\alias{paths}
\alias{print.paths}
\title{Causal Paths Analysis}
\usage{
paths(
  a,
  y,
  m,
  models,
  ps_model = NULL,
  data,
  nboot = 500,
  conf_level = 0.95,
  ...
)

\method{print}{paths}(x, digits = 3, ...)
}
\arguments{
\item{a}{a character string indicating the name of the treatment variable. The treatment
should be a binary variable taking either 0 or 1.}

\item{y}{a character string indicating the name of the outcome variable.}

\item{m}{a list of \eqn{K} character vectors indicating the names of \eqn{K} causally ordered mediators
\eqn{M_1,\ldots, M_K}.}

\item{models}{a list of \eqn{K+1} fitted model objects describing how the outcome depends on treatment,
  pretreatment confounders, and varying sets of mediators, where \eqn{K} is the number of mediators.
  \itemize{
  \item the first element is a baseline model of the outcome conditional on treatment and pretreatment
  confounders.
  \item the \eqn{k}th element is an outcome model conditional on treatment, pretreatment confounders,
  and mediators \eqn{M_1,\ldots, M_{k-1}}.
  \item the last element is an outcome model conditional on treatment, pretreatment confounders,
  and all of the mediators, i.e., \eqn{M_1,\ldots, M_K}.
  }

  The fitted model objects can be of type \code{\link{lm}}, \code{\link{glm}}, \code{\link[gbm]{gbm}},
  \code{\link[BART]{wbart}}, or \code{\link[BART]{pbart}}.}

\item{ps_model}{an optional propensity score model for treatment. It can be of type \code{\link{glm}},
\code{\link[gbm]{gbm}}, \code{\link[twang]{ps}}, or \code{\link[BART]{pbart}}. When it is provided,
the imputation-based weighting estimator is also used to compute path-specific causal effects.}

\item{data}{a data frame containing all variables.}

\item{nboot}{number of bootstrap iterations for estimating confidence intervals. Default is 500.}

\item{conf_level}{the confidence level of the returned two-sided confidence
intervals. Default is \code{0.95}.}

\item{...}{additional arguments to be passed to \code{boot::boot}, e.g.
\code{parallel} and \code{ncpus}. For the \code{print} method, additional arguments to be passed to
\code{print.default}}

\item{x}{a fitted model object returned by the \code{\link{paths}} function.}

\item{digits}{minimal number of significant digits printed.}
}
\value{
An object of class \code{paths}, which is a list containing the
  following elements \describe{
  \item{pure}{estimates of direct and path-specific effects via \eqn{M_1, \ldots, M_K}
    based on the pure imputation estimator.}
  \item{hybrid}{estimates of direct and path-specific effects via \eqn{M_1, \ldots, M_K}
    based on the imputation-based weighting estimator.}
  \item{varnames}{a list of character strings indicating the names of the pretreatment confounders (\eqn{X}),
  treatment(\eqn{A}), mediators (\eqn{M_1, \ldots, M_K}), and outcome (\eqn{Y}).}
  \item{formulas}{formulas for the outcome models.}
  \item{classes}{classes of the outcome models.}
  \item{families}{model families of the outcome models.}
  \item{args}{a list containing arguments of the outcome models.}
  \item{ps_formula}{formula for the propensity score model.}
  \item{ps_class}{class of the propensity score model.}
  \item{ps_family}{model family of the propensity score model.}
  \item{ps_args}{arguments of the propensity score model.}
  \item{data}{the original data.}
  \item{nboot}{number of bootstrap iterations.}
  \item{conf_level}{confidence level for confidence intervals.}
  \item{boot_out}{output matrix from the bootstrap iterations.}
  \item{call}{the matched call to the \code{paths} function.}
  }
}
\description{
\code{paths} estimates path-specific causal effects in the presence of \eqn{K(\geq 1)} causally
ordered mediators. It implements the pure imputation estimator and the imputation-based weighting
estimator (when a propensity score model is provided) as detailed in Zhou and Yamamoto (2020).
The user supplies the names of the treatment, outcome, mediator variables, \eqn{K+1} fitted models
characterizing the conditional mean of the outcome given treatment, pretreatment confounders, and
varying sets of mediators, and a data frame containing all the variables. The function returns
\eqn{K+1} path-specific causal effects that together constitute the total treatment effect.
When \eqn{K=1}, the path-specific causal effects are identical to the natural direct and indirect
effects in standard causal mediation analysis.
}
\examples{
data(tatar)

m1 <- c("trust_g1", "victim_g1", "fear_g1")
m2 <- c("trust_g2", "victim_g2", "fear_g2")
m3 <- c("trust_g3", "victim_g3", "fear_g3")
mediators <- list(m1, m2, m3)

formula_m0 <- annex ~ kulak + prosoviet_pre + religiosity_pre + land_pre +
  orchard_pre + animals_pre + carriage_pre + otherprop_pre + violence
formula_m1 <- update(formula_m0,    ~ . + trust_g1 + victim_g1 + fear_g1)
formula_m2 <- update(formula_m1,    ~ . + trust_g2 + victim_g2 + fear_g2)
formula_m3 <- update(formula_m2,    ~ . + trust_g3 + victim_g3 + fear_g3)
formula_ps <- violence ~ kulak + prosoviet_pre + religiosity_pre +
  land_pre + orchard_pre + animals_pre + carriage_pre + otherprop_pre

####################################################
# Causal Paths Analysis using GLM
####################################################

# outcome models
glm_m0 <- glm(formula_m0, family = binomial("logit"), data = tatar)
glm_m1 <- glm(formula_m1, family = binomial("logit"), data = tatar)
glm_m2 <- glm(formula_m2, family = binomial("logit"), data = tatar)
glm_m3 <- glm(formula_m3, family = binomial("logit"), data = tatar)
glm_ymodels <- list(glm_m0, glm_m1, glm_m2, glm_m3)

# propensity score model
glm_ps <- glm(formula_ps, family = binomial("logit"), data = tatar)

# causal paths analysis using glm
# note: For illustration purposes only a small number of bootstrap replicates are used
paths_glm <- paths(a = "violence", y = "annex", m = mediators,
  glm_ymodels, ps_model = glm_ps, data = tatar, nboot = 3)


####################################################
# Causal Paths Analysis using GBM
####################################################

require(gbm)

# outcome models
gbm_m0 <- gbm(formula_m0, data = tatar, distribution = "bernoulli", interaction.depth = 3)
gbm_m1 <- gbm(formula_m1, data = tatar, distribution = "bernoulli", interaction.depth = 3)
gbm_m2 <- gbm(formula_m2, data = tatar, distribution = "bernoulli", interaction.depth = 3)
gbm_m3 <- gbm(formula_m3, data = tatar, distribution = "bernoulli", interaction.depth = 3)
gbm_ymodels <- list(gbm_m0, gbm_m1, gbm_m2, gbm_m3)

# propensity score model via gbm
gbm_ps <- gbm(formula_ps, data = tatar, distribution = "bernoulli", interaction.depth = 3)

# causal paths analysis using gbm
# note: For illustration purposes only a small number of bootstrap replicates are used
paths_gbm <- paths(a = "violence", y = "annex", m = mediators,
  gbm_ymodels, ps_model = gbm_ps, data = tatar, nboot = 3)

}
\references{
Zhou, Xiang and Teppei Yamamoto. 2020. "\href{https://osf.io/2rx6p}{Tracing Causal Paths from Experimental and Observational Data}".
}
\seealso{
\code{\link{summary.paths}}, \code{\link{plot.paths}}, \code{\link{sens}}
}
