
## Overview

`econullnetr` is a package of functions for analysing ecological
networks (e.g. food webs, flower-visitation networks) using null model
analysis. The observed network is compared to networks generated by a
null model, which assumes that the frequency with which species (nodes
in the network) interact with one another is simply a consequence of
their relative abundance (i.e. how often they encounter one another).
Differences in the structure of observed and simulated networks (e.g. in
the interaction strengths between nodes) implies that other mechanisms
are involved in structuring the network.

`econullnetr` has seven functions:

1.  `generate_null_net()` to run null models.
2.  `test_interactions()` for comparing observed and modelled
    interactions between species.
3.  `plot_preferences()` for comparing a consumer species’ observed and
    expected interactions with the resource species.
4.  `bipartite_stats()` for calculating a broad set of statistics for
    bipartite networks to compare observed and modelled networks. This
    draws on the functions from the `bipartite` package.
5.  `plot_bipartite()` which is a wrapper for the `bipartite` package’s
    `plotweb()` function, superimposing the null model results.
6.  `generate_edgelist()` to export the null modelling results in a
    standard format for use with other R packages.
7.  `expand_matrix()` to convert summarised interaction matrices
    (e.g. as used by the `bipartite` package) into a format suitable
    for `generate_null_net`.

The package also includes three examples, each comprising multiple data
sets. See the package vignette and help files for full descriptions of
the package’s functionality and data sets.

## Usage

The following example runs a simple null model using one of the example
data sets from `econullnetr` and displays an example of
`bipartite_stats()` output and part of the output table for
`test_interactions()` (shortened for brevity). Full examples covering
all of the main functionality are described in the help files and
package vignette.

``` r
library(econullnetr)
set.seed(1234)
sil.null <- generate_null_net(Silene[, 2:7], Silene.plants[, 2:6], sims = 10,
                             c.samples = Silene[, 1],
                             r.samples = Silene.plants[, 1], prog.count = FALSE)


# Network-level statistics
bipartite_stats(sil.null, index.type = "networklevel",
                indices = c("linkage density", "weighted connectance", 
                            "interaction evenness"), intereven = "sum", 
                prog.count = FALSE)
```

    ##                       Observed      Null  Lower.CL  Upper.CL  Test       SES
    ## linkage density      5.0960242 6.8303241 6.4566289 7.1999193 Lower -6.740767
    ## weighted connectance 0.1415562 0.1902469 0.1804936 0.1999978 Lower -7.412841
    ## interaction evenness 0.8489881 0.8991247 0.8765239 0.9104503 Lower -4.482613

``` r
# First 10 rows of the output table for inter-specific (inter-node) interactions.
#   When running this code, two warnings will normally be generated to highlight
#   that: i) a very small number of model iterations was used for this example 
#   and ii) there is a large number (155) of individual tests in the full 
#   test_interactions table, so the risk of Type I errors needs to be considered.


test_interactions(sil.null, 0.95)[1:10, ]
```

    ##           Consumer             Resource Observed Null Lower.95.CL Upper.95.CL     Test        SES
    ## 1          big.fly Achillea.millefolium        1  0.1           0       0.775 Stronger  2.8460499
    ## 2          big.fly   Hypericum.pulchrum        0  0.3           0       1.000       ns -0.6210590
    ## 3          big.fly       Papaver.rhoeas        0  0.0           0       0.000       ns         NA
    ## 4          big.fly     Senecio.jacobaea        0  0.6           0       1.000       ns -1.1618950
    ## 5          big.fly       Silene.gallica        0  0.0           0       0.000       ns         NA
    ## 6  Bombus.pratorum Achillea.millefolium        0  0.0           0       0.000       ns         NA
    ## 7  Bombus.pratorum   Hypericum.pulchrum        0  0.1           0       0.775       ns -0.3162278
    ## 8  Bombus.pratorum       Papaver.rhoeas        0  0.0           0       0.000       ns         NA
    ## 9  Bombus.pratorum     Senecio.jacobaea        1  0.8           0       1.000       ns  0.4743416
    ## 10 Bombus.pratorum       Silene.gallica        0  0.1           0       0.775       ns -0.3162278

## Installation

The development version of `econullnetr` can be installed from GitHub:

``` r
# install.packages("devtools")
devtools::install_github("ivaughan/econullnetr")
```
