% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitted_draws.R, R/predicted_draws.R
\name{add_fitted_draws}
\alias{add_fitted_draws}
\alias{fitted_draws}
\alias{fitted_draws.default}
\alias{fitted_draws.stanreg}
\alias{fitted_draws.brmsfit}
\alias{add_linpred_draws}
\alias{linpred_draws}
\alias{add_predicted_draws}
\alias{predicted_draws}
\alias{predicted_draws.default}
\alias{predicted_draws.stanreg}
\alias{predicted_draws.brmsfit}
\title{Add draws from (possibly transformed) posterior linear predictors (the "fit") or from posterior predictions
of a model to a data frame}
\usage{
add_fitted_draws(newdata, model, value = ".value", ..., n = NULL,
  seed = NULL, re_formula = NULL, category = ".category",
  dpar = FALSE, scale = c("response", "linear"))

fitted_draws(model, newdata, value = ".value", ..., n = NULL,
  seed = NULL, re_formula = NULL, category = ".category",
  dpar = FALSE, scale = c("response", "linear"))

\method{fitted_draws}{default}(model, newdata, ...)

\method{fitted_draws}{stanreg}(model, newdata, value = ".value", ...,
  n = NULL, seed = NULL, re_formula = NULL, category = ".category",
  dpar = FALSE, scale = c("response", "linear"))

\method{fitted_draws}{brmsfit}(model, newdata, value = ".value", ...,
  n = NULL, seed = NULL, re_formula = NULL, category = ".category",
  dpar = FALSE, scale = c("response", "linear"))

add_linpred_draws(newdata, model, value = ".value", ..., n = NULL,
  seed = NULL, re_formula = NULL, category = ".category",
  dpar = FALSE, scale = c("response", "linear"))

linpred_draws(model, newdata, value = ".value", ..., n = NULL,
  seed = NULL, re_formula = NULL, category = ".category",
  dpar = FALSE, scale = c("response", "linear"))

add_predicted_draws(newdata, model, prediction = ".prediction", ...,
  n = NULL, seed = NULL, re_formula = NULL)

predicted_draws(model, newdata, prediction = ".prediction", ...,
  n = NULL, seed = NULL, re_formula = NULL)

\method{predicted_draws}{default}(model, newdata, ...)

\method{predicted_draws}{stanreg}(model, newdata,
  prediction = ".prediction", ..., n = NULL, seed = NULL,
  re_formula = NULL)

\method{predicted_draws}{brmsfit}(model, newdata,
  prediction = ".prediction", ..., n = NULL, seed = NULL,
  re_formula = NULL)
}
\arguments{
\item{newdata}{Data frame to generate predictions from. If omitted, most model types will
generate predictions from the data used to fit the model.}

\item{model}{A supported Bayesian model fit that can provide fits and predictions. Supported models
are listed in the second section of \link{tidybayes-models}: \emph{Models Supporting Prediction}. While other
functions in this package (like \code{\link{spread_draws}}) support a wider range of models, to work with
\code{add_fitted_draws} and \code{add_predicted_draws} a model must provide an interface for generating
predictions, thus more generic Bayesian modeling interfaces like \code{runjags} and \code{rstan} are not directly
supported for these functions (only wrappers around those languages that provide predictions, like \code{rstanarm}
and \code{brm}, are supported here).}

\item{value}{The name of the output column for \code{fitted_draws}; default \code{".value"}.}

\item{...}{Additional arguments passed to the underlying prediction method for the type of
model given.}

\item{n}{The number of draws per prediction / fit to return, or \code{NULL} to return all draws.}

\item{seed}{A seed to use when subsampling draws (i.e. when \code{n} is not \code{NULL}).}

\item{re_formula}{formula containing group-level effects to be considered in the prediction.
If \code{NULL} (default), include all group-level effects; if \code{NA}, include no group-level effects.
Some model types (such as \code{\link[brms]{brm}} and \code{\link[rstanarm]{stanreg-objects}}) allow
marginalizing over grouping factors by specifying new levels of a factor in \code{newdata}. In the case of
\code{\link[brms]{brm}}, you must also pass \code{allow_new_levels = TRUE} here to include new levels (see
\code{\link[brms]{predict.brmsfit}}).}

\item{category}{For \emph{some} ordinal and multinomial models (notably, \code{\link[brms]{brm}} models but
\emph{not} \code{\link[rstanarm]{stan_polr}} models), multiple sets of rows will be returned per input row for
\code{fitted_draws}, one for each category. The \code{category} argument specifies the name of the column
to put the category names into in the resulting data frame (default: \code{".category"}). The fact that multiple
rows per response are returned only for some model types reflects the fact that tidybayes takes the approach of
tidying whatever output is given to us, and the output from different modeling functions differ on this point.
See \code{vignette("tidy-brms")} and \code{vignette("tidy-rstanarm")} for examples of dealing with output
from ordinal models using both approaches.}

\item{dpar}{For \code{fitted_draws} and \code{add_fitted_draws}: Should distributional regression
parameters be included in the output? Valid only for models that support distributional regression parameters,
such as submodels for variance parameters (as in \code{brm}). If \code{TRUE}, distributional regression
parameters are included in the output as additional columns named after each parameter
(alternative names can be provided using a list or named vector, e.g. \code{c(sigma.hat = "sigma")}
would output the \code{"sigma"} parameter from a model as a column named \code{"sigma.hat"}).
If \code{FALSE} (the default), distributional regression parameters are not included.}

\item{scale}{Either \code{"response"} or \code{"linear"}. If \code{"response"}, results are returned
on the scale of the response variable. If \code{"linear"}, fitted values are returned on the scale of
the linear predictor.}

\item{prediction}{The name of the output column for \code{predicted_draws}; default \code{".prediction"}.}
}
\value{
A data frame (actually, a \code{\link[tibble]{tibble}}) with a \code{.row} column (a
factor grouping rows from the input \code{newdata}), \code{.chain} column (the chain
each draw came from, or \code{NA} if the model does not provide chain information),
\code{.iteration} column (the iteration the draw came from, or \code{NA} if the model does
not provide iteration information), and a \code{.draw} column (a unique index corresponding to each draw
from the distribution). In addition, \code{fitted_draws} includes a column with its name specified by
the \code{value} argument (default is \code{.value}) containing draws from the (transformed) linear predictor,
and \code{predicted_draws} contains a \code{.prediction} column containing draws from the posterior predictive
distribution. For convenience, the resulting data frame comes grouped by the original input rows.
}
\description{
Given a data frame and a model, adds draws from the posterior "fit" (aka the linear/link-level predictor)
or from the posterior predictions of the model to the data frame in a long format.
}
\details{
\code{add_fitted_draws} adds draws from (possibly transformed) posterior linear predictors (or "link-level" predictors) to
the data. It corresponds to \code{\link[rstanarm]{posterior_linpred}} in \code{rstanarm} or
\code{\link[brms]{fitted.brmsfit}} in \code{brms}.

\code{add_predicted_draws} adds draws from posterior predictions to
the data. It corresponds to \code{\link[rstanarm]{posterior_predict}} in \code{rstanarm} or
\code{\link[brms]{predict.brmsfit}} in \code{brms}.

\code{add_fitted_draws} and \code{fitted_draws} are alternate spellings of the
same function with opposite order of the first two arguments to facilitate use in data
processing pipelines that start either with a data frame or a model. Similarly,
\code{add_predicted_draws} and \code{predicted_draws} are alternate spellings.

Given equal choice between the two, \code{add_fitted_draws} and \code{add_predicted_draws}
are the preferred spellings.

\code{add_linpred_draws} and \code{linpred_draws} are alternative spellings of \code{fitted_draws}
and \code{add_fitted_draws} for consistency with \code{rstanarm} terminology (specifically
\code{\link[rstanarm]{posterior_linpred}}).
}
\examples{
\donttest{

library(ggplot2)
library(dplyr)
library(rstanarm)
library(modelr)

theme_set(theme_light())

m_mpg = stan_glm(mpg ~ hp * cyl, data = mtcars,
  # 1 chain / few iterations just so example runs quickly
  # do not use in practice
  chains = 1, iter = 500)

# draw 100 fit lines from the posterior and overplot them
mtcars \%>\%
  group_by(cyl) \%>\%
  data_grid(hp = seq_range(hp, n = 101)) \%>\%
  add_fitted_draws(m_mpg, n = 100) \%>\%
  ggplot(aes(x = hp, y = mpg, color = ordered(cyl))) +
  geom_line(aes(y = .value, group = paste(cyl, .draw)), alpha = 0.25) +
  geom_point(data = mtcars)

# plot posterior predictive intervals
mtcars \%>\%
  group_by(cyl) \%>\%
  data_grid(hp = seq_range(hp, n = 101)) \%>\%
  add_predicted_draws(m_mpg) \%>\%
  ggplot(aes(x = hp, y = mpg, color = ordered(cyl))) +
  stat_lineribbon(aes(y = .prediction), .width = c(.99, .95, .8, .5), alpha = 0.25) +
  geom_point(data = mtcars) +
  scale_fill_brewer(palette = "Greys")

}
}
\seealso{
\code{\link{spread_draws}}
}
\author{
Matthew Kay
}
\keyword{manip}
