% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_Y_MVN_X.R
\name{sim_Y_MVN_X}
\alias{sim_Y_MVN_X}
\title{Simulate Scalar Outcomes from Simulated Spatially Correlated Predictors}
\usage{
sim_Y_MVN_X(
  N,
  B,
  L = NULL,
  R = NULL,
  S = NULL,
  Q = NULL,
  use.spam = TRUE,
  mu = 0,
  rand.err = 1,
  dist = "gaussian",
  incl.subjectID = TRUE,
  threshold.method = "none",
  Y.thresh = NULL,
  X.categorical = FALSE,
  X.num.categories = 2,
  X.category.type = "percentile",
  X.manual.thresh = NULL,
  X.cat.names = NULL,
  print.out = FALSE
)
}
\arguments{
\item{N}{The number of draws to take from MVN; i.e., the number of subjects.}

\item{B}{A vector parameter values; i.e. "betas". Note that \code{length(B)}
must equal \code{p + 1 = n.row * n.col + 1}; e.g. for normal outcomes
\eqn{Y = XB + e} with \code{Y} a scalar outcome and \code{e} the random error.}

\item{L}{\code{L} and \code{R} are lower and upper triangular matrices, respectively,
and are the Cholesky factor(s) of the desired covariance matrix for the MVN.
Obtain \code{L} or \code{R} via \code{chol_s2Dp()} with settings
\code{triangle = "lower"} or \code{triangle = "upper"}, respectively.
Specify either \code{L} or \code{R}, but NOT both.}

\item{R}{\code{L} and \code{R} are lower and upper triangular matrices, respectively,
and are the Cholesky factor(s) of the desired covariance matrix for the MVN.
Obtain \code{L} or \code{R} via \code{chol_s2Dp()} with settings
\code{triangle = "lower"} or \code{triangle = "upper"}, respectively.
Specify either \code{L} or \code{R}, but NOT both.}

\item{S}{A covariance or precision matrix respectively. These are for use with \code{spam},
and can be extracted from output of \code{\link[sim2Dpredictr]{chol_s2Dp}} after choosing
\code{return.cov = TRUE} or \code{return.prec = TRUE}, respectively.}

\item{Q}{A covariance or precision matrix respectively. These are for use with \code{spam},
and can be extracted from output of \code{\link[sim2Dpredictr]{chol_s2Dp}} after choosing
\code{return.cov = TRUE} or \code{return.prec = TRUE}, respectively.}

\item{use.spam}{Logical. If \code{use.spam = TRUE} then use tools from the R package \code{spam};
otherwise, base R functions are employed. For large dimension MVN with sparse correlation
structure, \code{spam} is recommended; otherwise, base R may be faster. Defaults to \code{FALSE}.
Requires either the covariance matrix \code{S} or precision matrix, \code{Q}, that corresponds to the
Cholesky factor.}

\item{mu}{One of the following:
\itemize{
   \item A single scalar value for common mean.
   \item A vector of length \code{nrow(R)} (equivalently \code{nrow(R)})
    containing means for the MVN.
}}

\item{rand.err}{A vector for the random error standard deviation when \code{dist = "gaussian"},
or thresholding is used to obtain non-Normal draws. Must have length 1 or length N.}

\item{dist}{The distribution of the scalar outcome.
\itemize{
    \item \code{dist = "gaussian"} has \eqn{Y = XB + e}, where
    \eqn{e ~ N(0, rand.err)}.
    \item \code{dist = "binomial"} is drawn from eqn{Bin(XB, XB(1-XB))}
    using \code{rbinom()} when \code{binary.method = "traditional"}. If
    \code{binary.method = "gaussian"}, then simulation is based on a
    cutoff using \code{binary.cutoff}.
    \item \code{dist = "poisson"} is drawn from \eqn{Poisson(XB)} using
    \code{rpois()}.
}}

\item{incl.subjectID}{When \code{incl.subjectID = TRUE} a column of subject indices
is generated.}

\item{threshold.method}{One of \code{"none", "manual", "percentile", "round"}.
When \code{"none"} draws from Binomial or Poisson distributions are taken subject-wise
using base \code{R} functions. For the remaining options, draws are first taken from a
Normal distribution and then thresholded. \code{"manual"} uses \code{Y.thresh} to manually
select a cutoff, \code{"percentile"} uses \code{Y.thresh} to select percentiles used to bin
outcomes, and \code{"round"} sets values equal or less than 0 to 0, and rounds all positive
values to the nearest whole number.}

\item{Y.thresh}{A manual value used to threshold when \code{threshold.method = "manual"}; values
equal or greater than the cutoff are assigned 1 and all others 0. When \code{threshold.method = "percentile"},
a percentile to use to bin outcomes.}

\item{X.categorical}{Default is \code{X.categorical = FALSE}. If
\code{X.categorical = TRUE} then thresholds are applied to categorize
each predictor/image value.}

\item{X.num.categories}{A scalar value denoting the number of categories
in which to divide the data.}

\item{X.category.type}{Tells R how to categorize the data. Options are
\code{X.category.type = c("percentile", "manual")}.
If \code{X.category.type = "percentile"} then the data are divided into
percentiles based on \code{X.num.categories}; e.g. if \code{X.num.categories = 4}
then the values are divided into quartiles, and values in Q1 equal 0, between Q1
and Q2 equal 1, between Q2 and Q3 equal 2, and greater than Q3 equal 3.
If \code{X.category.type = "manual"} then specify the cutoff points with
\code{X.manual.thresh}.}

\item{X.manual.thresh}{A vector containing the thresholds for categorizing the
values; e.g. if \code{X.num.categories = 4} and \code{X.manual.thresh = c(-3, 1, 17)},
then values less than -3 are set to 0, equal or greater than -3 and less than 1
are set to 1, equal or greater than 1 but less than 17 are set to 2, and equal or
greater than 17 are set to 3. Note that \code{length(X.manual.thresh)} must always
equal \code{X.num.categories - 1}.}

\item{X.cat.names}{A vector of category names. If \code{X.cat.names = NULL} then
then the initial integers assigned are left as the values; the names in
\code{X.cat.names} are assigned in ascending order.}

\item{print.out}{If \code{print.out = TRUE} then print the following for
 each subject, indexed y: \itemize{
     \item \code{X[y] \%*\% B}
     \item \code{p[y]}, \code{lambda[y]} for Binomial, Poisson, respectively.
 }
This is useful to see the effect of image parameter selection and beta
parameter selection on distributional parameters for the outcome of interest.}
}
\value{
A data frame where each row consists of a single subject's data.
Col 1 is the outcome, Y, and each successive column contains the subject
predictor values.
}
\description{
N spatially correlated design vectors are simulated from an MVN. These
design vectors are used to then simulate scalar outcomes that have
one of Gaussian, Binomial, or Poisson distributions.
}
\note{
Careful parameter selection, i.e. \code{B}, is necessary to ensure that
simulated outcomes are reasonable; in particular, counts arising from the Poisson
distribution can be unnaturally large.
}
\examples{
## generate precision matrix and take Cholesky decomposition
Rpre <- chol_s2Dp(im.res = c(5, 5), matrix.type = "prec",
                  use.spam = TRUE, neighborhood = "ar1",
                  triangle = "upper", return.prec = TRUE)
## Generate correlation matrix & take Cholesky decomposition
Rcov <- chol_s2Dp(corr.structure = "ar1", im.res = c(5, 5), rho = 0.5,
                  triangle = "upper",
                  use.spam = FALSE, neighborhood = "none")

## Define non-zero beta values
Bex <- beta_builder(row.index = c(3, 3, 4, 4), col.index = c(3, 4, 3, 4),
                    im.res = c(5, 5),
                    B0 = 0, B.values = rep(1, 4),
                    output.indices = FALSE)
## Simulate Datasets
## parameter values
Nex = 100
set.seed(28743)

## with precision matrix
Gauss.exp <- sim_Y_MVN_X(N = Nex, B = Bex,
                         R = Rpre$R, Q = Rpre$Q,
                         dist = "gaussian")
hist(Gauss.exp$Y)

## with covariance matrix
Gauss.exc <- sim_Y_MVN_X(N = Nex, B = Bex,
                         R = Rcov$R, S = Rcov$S,
                         dist = "gaussian")
hist(Gauss.exc$Y)

## direct draws from binomial
Bin.ex <- sim_Y_MVN_X(N = Nex, B = Bex, R = Rcov$R, S = Rcov$S,
                      dist = "binomial", print.out = TRUE)
table(Bin.ex$Y)

## manual cutoff
Bin.ex2 <- sim_Y_MVN_X(N = Nex, B = Bex,
                       R = Rcov$R, S = Rcov$S,
                       dist = "binomial",
                       threshold.method = "manual",
                       Y.thresh = 1.25)
table(Bin.ex2$Y)

## percentile cutoff
Bin.ex3 <- sim_Y_MVN_X(N = Nex, B = Bex,
                       R = Rcov$R, S = Rcov$S,
                       dist = "binomial",
                       threshold.method = "percentile",
                       Y.thresh = 0.75)
table(Bin.ex3$Y)

## Poisson Example - note the large counts
Pois.ex <- sim_Y_MVN_X(N = Nex, B = Bex,
                       R = Rcov$R, S = Rcov$S,
                       dist = "poisson", print.out = TRUE)
mean(Pois.ex$Y)
quantile(Pois.ex$Y, probs = c(0, 0.1, 0.25, 0.45, 0.5, 0.75, 0.9, 0.95, 0.99, 1))
hist(Pois.ex$Y)
}
\references{
\insertRef{spam}{sim2Dpredictr}

\insertRef{Ripley:1987}{sim2Dpredictr}

\insertRef{Rue:2001}{sim2Dpredictr}
}
