#' @name importArms
#' @title Import Study Arm Names
#' 
#' @description Import Arms into a project or to rename existing Arms in a 
#'   project. You may use the parameter \code{override = TRUE} as a 'delete 
#'   all + import' action in order to erase all existing Arms in the 
#'   project while importing new Arms. Notice: Because of the 'override' 
#'   parameter's destructive nature, this method may only use 
#'   \code{override = TRUE} for projects in Development status.
#'   
#' @param rcon A REDCap connection object as generated by 
#'   \code{redcapConnection}.
#' @param arms_data A \code{data.frame} with two columns.  The first column 
#'   is an integer-like value with the name \code{arm_num}. The second is
#'   a character value with the name \code{name}.
#' @param override \code{logical(1)}. When \code{TRUE}, the action is to 
#'   delete all of the arms in the project and import the contents of 
#'   \code{arms_data}.  The default setting is \code{FALSE}, which only
#'   allows arms to be renamed or added.
#' @param refresh \code{logical(1)}. If \code{TRUE}, the cached data will
#'   be refreshed after the import.
#' @param ... additional arguments to pass to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @section REDCap API Documentation:
#' This method allows you to import Arms into a project or to rename 
#' existing Arms in a project. You may use the parameter override=1 as a 
#' 'delete all + import' action in order to erase all existing Arms in the 
#' project while importing new Arms. Notice: Because of the 'override' 
#' parameter's destructive nature, this method may only use override=1 
#' for projects in Development status.
#' 
#' NOTE: This only works for longitudinal projects.
#' 
#' @section REDCap Version:
#' At least 8.1.17+ (and likely some earlier versions)
#' 
#' @return 
#' No value is returned.
#' 
#' @references
#' Please refer to your institution's API documentation.
#' 
#' @export

importArms <- function(rcon, 
                       arms_data, 
                       override = FALSE, 
                       ...){
  UseMethod("importArms")
}

#' @rdname importArms
#' @export

importArms.redcapApiConnection <- function(rcon, 
                                           arms_data, 
                                           override       = FALSE, 
                                           refresh        = TRUE,
                                           ...,
                                           error_handling = getOption("redcap_error_handling"), 
                                           config         = list(), 
                                           api_param      = list()){
  
  ##################################################################
  # Argument Validation 
  
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_data_frame(x = arms_data,
                               ncols = 2,
                               add = coll)
  
  checkmate::assert_logical(x = override,
                            len = 1,
                            add = coll)
  
  checkmate::assert_logical(x = refresh, 
                            len = 1, 
                            add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling,
                                        choices = c("null", "error"),
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  checkmate::assert_subset(x = names(arms_data),
                           choices = c("arm_num", "name"),
                           add = coll)
  
  checkmate::reportAssertions(coll)
  
  checkmate::assert_integerish(arms_data[["arm_num"]],
                               add = coll,
                               .var.name = "arms_data$arm_num")
  
  checkmate::assert_character(arms_data[["name"]],
                              add = coll,
                              .var.name = "arms_data$name")
  
  checkmate::reportAssertions(coll)
      
   ##################################################################
  # Make API Body List
  
  body <- list(token = rcon$token,
               content = "arm",
               override = as.numeric(override),
               action = "import",
               format = "csv",
               data = writeDataForImport(arms_data))
  
  body <- body[lengths(body) > 0]
  
   ##################################################################
  # API Call
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200) return(redcap_error(response, error_handling))
  
  message(sprintf("Arms imported: %s", 
                  as.character(response)))
  
  if (refresh && rcon$has_arms()){
    rcon$refresh_arms()
    rcon$refresh_projectInformation()
  }
}
