#'Plots Cut Point from Gaussian Mixture Models
#'
#'Generates a plot of data densities with a cut point generated by a Gaussian mixture model
#'@usage plot_cut_point(m, plot=TRUE)
#'@param m An object of class \code{mixEM} corresponding with the fit GMM
#'@param plot A logical argument for generating the plot. If FALSE, only summary output from the GMM is produced. If TRUE, a plot with the overlaid cut point is produced. Default is set to TRUE.
#'@details Gaussian mixture models are often used to derive cut points, or lines of separation between clusters in feature space (See Benaglia et al. 2009 for more). The plot_cut_point function plots data densities with the overlaid cut point (the mean of the calculated \code{mu}) from \code{mixEM} objects, which are GMM's fit using the \code{mixtools} package.
#'@examples
#'mixmdl <- mixtools::normalmixEM(faithful$waiting, k = 2)
#'
#'plot_cut_point(mixmdl, plot = TRUE) # produces a plot with the cut point
#'plot_cut_point(mixmdl, plot = FALSE) # produces only a summary of the GMM
#'@references Benaglia, T., Chauveau, D., Hunter, D. and Young, D., 2009. mixtools: An R package for analyzing finite mixture models. Journal of Statistical Software, 32(6), pp.1-29.
#'
#'@export

plot_cut_point <- function(m, plot=TRUE) {
  if (!requireNamespace("mixtools", quietly = TRUE)) {
    stop("Package \"mixtools\" needed for this function to work. Please install it.",
         call. = FALSE)
  }
  if (!requireNamespace("methods", quietly = TRUE)) {
    stop("Package \"methods\" needed for this function to work. Please install it.",
         call. = FALSE)
  }
  m <- try(methods::as(m, "mixEM", strict=TRUE))
  if (!inherits(m, "mixEM")){
    stop("must be a mixEM object", call. = FALSE)
  }
  x <- m$x
  if (plot == TRUE){
    mm <- mixtools::normalmixEM(x); graphics::hist(x, ylab = "Density", xlab = "x",
                                         main = "Cutpoint from Gaussian Mixture Model",
                                         col = "lightgray", border = "darkgray"); graphics::grid(); graphics::abline(v = mean(mm$mu))
  } else summary(mixtools::normalmixEM(x))
}
