#' Print and plot methods
#'
#' \code{print} and \code{plot} methods for objects of class \code{'loo'} and
#' \code{print} method for objects of class \code{'compare.loo'}.
#'
#' @export
#' @param x A list with class \code{'loo'} (as returned by the \code{\link{loo}}
#'   function or, for \code{print} only, the \code{\link{waic}} function). For
#'   \code{print}, \code{x} can also have class \code{'compare.loo'} (as
#'   returned by \code{\link{compare}}).
#' @param ... For \code{plot}, arguments to pass to \code{\link[graphics]{text}}
#'   if \code{label_points = TRUE}.
#' @param digits An integer passed to \code{\link[base]{round}}.
#' @param warn Logical. If \code{TRUE} (the default), a warning message will be
#'   printed if any estimates of the Pareto shape parameter \eqn{k} are
#'   problematic. See the PSIS-LOO section in \code{\link{loo-package}} for
#'   details on the interpretation of \eqn{k}. Ignored if \code{x} was generated
#'   by \code{\link{waic}}.
#' @param plot_k Logical. If \code{TRUE} the estimates of the Pareto shape
#'   parameter \eqn{k} are plotted. Ignored if \code{x} was generated by
#'   \code{\link{waic}}. To just plot \eqn{k} without printing use
#'   \code{plot(x)}.
#' @param label_points Logical. If \code{TRUE} the observation numbers
#'   corresponding to any values of \eqn{k} greater than 0.5 will be displayed
#'   in the plot. Any arguments specified in \code{...} will be passed to
#'   \code{\link[graphics]{text}} and can be used to control the appearance of
#'   the labels.
#'
#' @return The \code{print} methods return \code{x} invisibly. The \code{plot}
#'   method is called for its side effect and does not return anything. If
#'   \code{x} is the result of a call to \code{\link{loo}}, \code{plot(x)}
#'   produces a plot of the estimates of the Pareto shape parameter \eqn{k}.
#'   There is no \code{plot} method for objects generated by a call to
#'   \code{\link{waic}}.
#'
print.loo <- function(x, ..., digits = 1, warn = TRUE, plot_k = FALSE) {
  lldims <- paste(attr(x, "log_lik_dim"), collapse = " by ")
  cat("Computed from", lldims, "log-likelihood matrix\n\n")
  z <- x[-grep("pointwise|pareto_k", names(x))]
  uz <- unlist(z)
  nms <- names(uz)
  ses <- grepl("se", nms)
  out <- data.frame(Estimate = uz[!ses], SE = uz[ses])
  print(.fr(out, digits), quote = FALSE)
  if ("pareto_k" %in% names(x)) {
    if (warn) k_warnings(x$pareto_k, digits)
    if (plot_k) plot(x, ...)
  }
  if (warn && "p_waic" %in% colnames(x[["pointwise"]])) {
   pwaic_warnings(x$pointwise[, "p_waic"], digits)
  }
  invisible(x)
}

#' @rdname print.loo
#' @export
print.compare.loo <- function(x, ..., digits = 1) {
  print(.fr(x, digits), quote = FALSE)
  invisible(x)
}

#' @rdname print.loo
#' @export
plot.loo <- function(x, ..., label_points = FALSE) {
  if (is.null(x$pareto_k)) stop("No Pareto k values found.")
  k <- x$pareto_k
  k_inf <- !is.finite(k)
  if (any(k_inf)) {
    warning(signif(100 * mean(k_inf), 2),
            "% of Pareto k estimates are Inf/NA/NaN and not plotted.")
  }
  plot_k(k, ..., label_points = label_points)
}
