% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eta_squared.R
\name{eta_squared}
\alias{eta_squared}
\alias{omega_squared}
\alias{epsilon_squared}
\alias{cohens_f}
\title{Effect size for ANOVA}
\usage{
eta_squared(model, partial = TRUE, ci = 0.9, ...)

omega_squared(model, partial = TRUE, ci = 0.9, ...)

epsilon_squared(model, partial = TRUE, ci = 0.9, ...)

cohens_f(model, partial = TRUE, ci = 0.9, ...)
}
\arguments{
\item{model}{A model, ANOVA object, or the result of \code{parameters::model_parameters}.}

\item{partial}{If \code{TRUE}, return partial indices.}

\item{ci}{Confidence Interval (CI) level}

\item{...}{Arguments passed to or from other methods (ignored).}
}
\value{
A data frame with the effect size(s) and confidence interval(s).

A data frame containing the effect size values and their confidence intervals.
}
\description{
Functions to compute effect size measures for ANOVAs, such as Eta, Omega and Epsilon squared,
and Cohen's f (or their partialled versions) for \code{aov}, \code{aovlist} and \code{anova}
models. These indices represent an estimate of how much variance in the response variables
is accounted for by the explanatory variable(s).
\cr\cr
Effect sizes are computed using the sums of squares obtained from \code{anova(model)} which
might not always be appropriate (\strong{\emph{Yeah... ANOVAs are hard...}}). See details.
}
\details{
For \code{aov} and \code{aovlist} models, the effect sizes are computed directly with
Sums-of-Squares. For all other model, the model is passed to \code{anova()}, and effect
sizes are approximated via test statistic conversion (see \code{\link{F_to_eta2} for
more details.})

\subsection{Type of Sums of Squares}{
The sums of squares (or F statistics) used for the computation of the effect sizes is
based on those returned by \code{anova(model)} (whatever those may be - for \code{aov}
and \code{aovlist} these are \emph{type-1} sums of squares; for \code{merMod} these are
\emph{type-3} sums of squares). Make sure these are the sums of squares you are intrested
in (you might want to pass the result of \code{car::Anova(mode, type = 3)}).
\cr\cr
It is generally recommended to fit models with \emph{\code{contr.sum} factor weights} and
\emph{centered covariates}, for sensible results. See examples.
}

\subsection{Confidence Intervals}{
Confidence intervals are estimated using the Noncentrality parameter method;
These methods searches for a the best \code{ncp} (non-central parameters) for
of the noncentral F distribution for the desired tail-probabilities,
and then convert these \code{ncp}s to the corresponding effect sizes.
}

\subsection{Omega Squared}{
Omega squared is considered as a lesser biased alternative to eta-squared, especially
when sample sizes are small (Albers \& Lakens, 2018). Field (2013) suggests the following
interpretation heuristics:
\itemize{
  \item Omega Squared = 0 - 0.01: Very small
  \item Omega Squared = 0.01 - 0.06: Small
  \item Omega Squared = 0.06 - 0.14: Medium
  \item Omega Squared > 0.14: Large
}

} \subsection{Epsilon Squared}{
It is one of the least common measures of effect sizes: omega squared and eta squared are
used more frequently. Although having a different name and a formula in appearance
different, this index is equivalent to the adjusted R2 (Allen, 2017, p. 382).

} \subsection{Cohen's f}{
Cohen's f can take on values between zero, when the population
 means are all equal, and an indefinitely large number as standard deviation of means
 increases relative to the average standard deviation within each group. Cohen has
 suggested that the values of 0.10, 0.25, and 0.40 represent small, medium, and large
 effect sizes, respectively.
}
}
\examples{
\donttest{
library(effectsize)
mtcars$am_f <- factor(mtcars$am)
mtcars$cyl_f <- factor(mtcars$cyl)

model <- aov(mpg ~ am_f * cyl_f, data = mtcars)

eta_squared(model)
omega_squared(model)
epsilon_squared(model)
cohens_f(model)
(etas <- eta_squared(model, partial = FALSE))

if(require(see)) plot(etas)


model <- aov(mpg ~ cyl_f * am_f + Error(vs / am_f), data = mtcars)
epsilon_squared(model)

# Recommended:
# Type-3 effect sizes + effects coding
if (require(car, quietly = TRUE)) {
  contrasts(mtcars$am_f) <- contr.sum
  contrasts(mtcars$cyl_f) <- contr.sum

  model <- aov(mpg ~ am_f * cyl_f, data = mtcars)
  model_anova <- car::Anova(model, type = 3)

  eta_squared(model_anova)
}

if (require("parameters")) {
  data(mtcars)
  model <- lm(mpg ~ wt + cyl, data = mtcars)
  mp <- model_parameters(model)
  eta_squared(mp)
}

if (require(lmerTest, quietly = TRUE)) {
  model <- lmer(mpg ~ am_f * cyl_f + (1|vs), data = mtcars)
  omega_squared(model)
}
}
}
\references{
\itemize{
 \item Albers, C., \& Lakens, D. (2018). When power analyses based on pilot data are biased: Inaccurate effect size estimators and follow-up bias. Journal of experimental social psychology, 74, 187-195.
 \item Allen, R. (2017). Statistics and Experimental Design for Psychologists: A Model Comparison Approach. World Scientific Publishing Company.
 \item Field, A. (2013). Discovering statistics using IBM SPSS statistics. sage.
 \item Kelley, K. (2007). Methods for the behavioral, educational, and social sciences: An R package. Behavior Research Methods, 39(4), 979-984.
 \item Kelley, T. (1935) An unbiased correlation ratio measure. Proceedings of the National Academy of Sciences. 21(9). 554-559.
}

The computation of CIs is based on the implementation done by Stanley (2018) in the \code{ApaTables} package and Kelley (2007) in the \code{MBESS} package. All credits go to them.
}
\seealso{
\code{\link{F_to_eta2}}
}
