% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/INMF.R
\name{onlineINMF}
\alias{onlineINMF}
\title{Perform Integrative Non-negative Matrix Factorization Using Online Learning}
\usage{
onlineINMF(
  objectList,
  newDatasets = NULL,
  project = FALSE,
  k = 20,
  lambda = 5,
  maxEpoch = 5,
  minibatchSize = 5000,
  maxHALSIter = 1,
  permuteChunkSize = 1000,
  nCores = 2,
  Hinit = NULL,
  Vinit = NULL,
  Winit = NULL,
  Ainit = NULL,
  Binit = NULL,
  verbose = FALSE
)
}
\arguments{
\item{objectList}{list of input datasets. List elements should all be of the
same class. Viable classes include: matrix, dgCMatrix,
\link{H5Mat}, \link{H5SpMat}.}

\item{newDatasets}{Same requirements as for new arriving datasets. Default
\code{NULL} for scenario 1, specify for scenario 2 or 3.}

\item{project}{Logical scalar, whether to run scenario 3. See description.
Default  \code{FALSE}.}

\item{k}{Integer. Inner dimensionality to factorize the datasets into.
Default \code{20}.}

\item{lambda}{Regularization parameter. Larger values penalize
dataset-specific effects more strongly (i.e. alignment should increase as
\code{lambda} increases). Default \code{5}.}

\item{maxEpoch}{The number of epochs to iterate through. Default \code{5}.}

\item{minibatchSize}{Total number of cells in each mini-batch. Default
\code{5000}.}

\item{maxHALSIter}{Maximum number of block coordinate descent (HALS
algorithm) iterations to perform for each update of \eqn{W} and \eqn{V}.
Default \code{1}. Changing this parameter is not recommended.}

\item{permuteChunkSize}{Number of cells in a chunk being shuffled before
subsetting to minibatches. Only appliable to in-memory data and for Scenario 
1 and 2. Default \code{1000}.}

\item{nCores}{The number of parallel tasks that will be spawned.
Default \code{2}}

\item{Hinit, Vinit, Winit, Ainit, Binit}{Pass the previous factorization result
for datasets existing in \code{objectList}, in order to run scenario 2 or 3.
All should have \code{length(objectList)} matrices inside. See description
for dimensionality of \eqn{H_i}, \eqn{V_i} and \eqn{W_i}. \eqn{A_i} should
be of size \eqn{k \times k} and \eqn{B_i} should be of size \eqn{m \times k}}

\item{verbose}{Logical scalar. Whether to show information and progress.
Default \code{FALSE}.}
}
\value{
A list of the following elements:
\itemize{
 \item{\code{H} - a list of result \eqn{H_i} matrices of size
 \eqn{n_i \times k}}
 \item{\code{V} - a list of result \eqn{V_i} matrices}
 \item{\code{W} - the result \eqn{W} matrix}
 \item{\code{A} - a list of result \eqn{A_i} matrices, \eqn{k \times k}}
 \item{\code{B} - a list of result \eqn{B_i} matrices, \eqn{m \times k}}
 \item{\code{objErr} - the final objective error value.}
}
}
\description{
Performs integrative non-negative matrix factorization (iNMF) (J.D. Welch,
2019, C. Gao, 2021) using online learning approach to return factorized
\eqn{H}, \eqn{W}, and \eqn{V} matrices. The objective function is stated as

\deqn{\arg\min_{H\ge0,W\ge0,V\ge0}\sum_{i}^{d}||E_i-(W+V_i)Hi||^2_F+
\lambda\sum_{i}^{d}||V_iH_i||_F^2}

where \eqn{E_i} is the input non-negative matrix of the \eqn{i}'th dataset,
\eqn{d} is the total number of datasets. \eqn{E_i} is of size
\eqn{m \times n_i} for \eqn{m} features and \eqn{n_i} sample points,
\eqn{H_i} is of size \eqn{k \times n_i}, \eqn{V_i} is of size
\eqn{m \times k}, and \eqn{W} is of size \eqn{m \times k}.

Different from \code{\link{inmf}} which optimizes the objective with ANLS
approach, \code{onlineINMF} optimizes the same objective with online learning
strategy, where it updates mini-batches of \eqn{H_i} solving the NNLS
problem, and updates \eqn{V_i} and \eqn{W} with HALS multiplicative method.

This function allows online learning in 3 scenarios:

\enumerate{
 \item Fully observed datasets;
 \item Iterative refinement using continually arriving datasets;
 \item Projection of new datasets without updating the existing factorization
}
}
\examples{
library(Matrix)

# Scenario 1 with sparse matrices
set.seed(1)
res1 <- onlineINMF(list(ctrl.sparse, stim.sparse),
                   minibatchSize = 50, k = 10, verbose = FALSE)

# Scenario 2 with H5 dense matrices
h5dense1 <- H5Mat(filename = system.file("extdata", "ctrl_dense.h5",
                             package = "RcppPlanc", mustWork = TRUE),
                                          dataPath = "scaleData")
h5dense2 <- H5Mat(filename = system.file("extdata", "stim_dense.h5",
                             package = "RcppPlanc", mustWork = TRUE),
                                          dataPath = "scaleData")
res2 <- onlineINMF(list(ctrl = h5dense1), minibatchSize = 50, k = 10, verbose = FALSE)
res3 <- onlineINMF(list(ctrl = h5dense1),
                   newDatasets = list(stim = h5dense2),
                   Hinit = res2$H, Vinit = res2$V, Winit = res2$W,
                   Ainit = res2$A, Binit = res2$B,
                   minibatchSize = 50, k = 10, verbose = FALSE)

# Scenario 3 with H5 sparse matrices
h5sparse1 <- H5SpMat(filename = system.file("extdata", "ctrl_sparse.h5",
                                package = "RcppPlanc", mustWork = TRUE),
                                valuePath = "scaleDataSparse/data",
                                rowindPath = "scaleDataSparse/indices",
                                colptrPath = "scaleDataSparse/indptr",
                                nrow = nrow(ctrl.sparse),
                                ncol = ncol(ctrl.sparse))
h5sparse2 <- H5SpMat(filename = system.file("extdata", "stim_sparse.h5",
                                package = "RcppPlanc", mustWork = TRUE),
                                valuePath = "scaleDataSparse/data",
                                rowindPath = "scaleDataSparse/indices",
                                colptrPath = "scaleDataSparse/indptr",
                                nrow = nrow(stim.sparse),
                                ncol = nrow(stim.sparse))
res4 <- onlineINMF(list(ctrl = h5sparse1), minibatchSize = 50, k = 10, verbose = FALSE)
res5 <- onlineINMF(list(ctrl = h5sparse1),
                   newDatasets = list(stim = h5sparse2), project = TRUE,
                   Hinit = res4$H, Vinit = res4$V, Winit = res4$W,
                   Ainit = res4$A, Binit = res4$B,
                   minibatchSize = 50, k = 10, verbose = FALSE)

}
\references{
Joshua D. Welch and et al., Single-Cell Multi-omic Integration
Compares and Contrasts Features of Brain Cell Identity, Cell, 2019

Chao Gao and et al., Iterative single-cell multi-omic integration using
online learning, Nat Biotechnol., 2021
}
\author{
Yichen Wang
}
