% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CST_RainFARM.R
\name{CST_RainFARM}
\alias{CST_RainFARM}
\title{RainFARM stochastic precipitation downscaling of a CSTools object}
\usage{
CST_RainFARM(
  data,
  nf,
  weights = 1,
  slope = 0,
  kmin = 1,
  nens = 1,
  fglob = FALSE,
  fsmooth = TRUE,
  nprocs = 1,
  time_dim = NULL,
  verbose = FALSE,
  drop_realization_dim = FALSE
)
}
\arguments{
\item{data}{An object of the class 's2dv_cube' as returned by `CST_Load`, 
containing the spatial precipitation fields to downscale.
The data object is expected to have an element named \code{$data} with at least two
spatial dimensions named "lon" and "lat" and one or more dimensions over which
to compute average spectral slopes (unless specified with parameter \code{slope}),
which can be specified by parameter \code{time_dim}.
The number of longitudes and latitudes in the input data is expected to be even and the same. If not
the function will perform a subsetting to ensure this condition.}

\item{nf}{Refinement factor for downscaling (the output resolution is increased by this factor).}

\item{weights}{Matrix with climatological weights which can be obtained using
the \code{CST_RFWeights} function. If \code{weights=1.} (default) no weights are used.
The matrix should have dimensions (lon, lat) in this order.
The names of these dimensions are not checked.}

\item{slope}{Prescribed spectral slope. The default is \code{slope=0.}
meaning that the slope is determined automatically over the dimensions specified by \code{time_dim}.}

\item{kmin}{First wavenumber for spectral slope (default: \code{kmin=1}).}

\item{nens}{Number of ensemble members to produce (default: \code{nens=1}).}

\item{fglob}{Logical to conserve global precipitation over the domain (default: FALSE).}

\item{fsmooth}{Logical to conserve precipitation with a smoothing kernel (default: TRUE).}

\item{nprocs}{The number of parallel processes to spawn for the use for parallel computation in multiple cores. (default: 1)}

\item{time_dim}{String or character array with name(s) of dimension(s)
(e.g. "ftime", "sdate", "member" ...) over which to compute spectral slopes.
If a character array of dimension names is provided, the spectral slopes
will be computed as an average over all elements belonging to those dimensions.
If omitted one of c("ftime", "sdate", "time") is searched and the first one with more
than one element is chosen.}

\item{verbose}{Logical for verbose output (default: FALSE).}

\item{drop_realization_dim}{Logical to remove the "realization" stochastic ensemble dimension, 
needed for saving data through function CST_SaveData (default: FALSE)
with the following behaviour if set to TRUE:

1) if \code{nens==1}: the dimension is dropped;

2) if \code{nens>1} and a "member" dimension exists:
   the "realization" and "member" dimensions are compacted (multiplied) and the resulting dimension is named "member";

3) if \code{nens>1} and a "member" dimension does not exist: the "realization" dimension is renamed to "member".}
}
\value{
CST_RainFARM() returns a downscaled CSTools object (i.e., of the 
class 's2dv_cube').
If \code{nens>1} an additional dimension named "realization" is added to the 
\code{$data} array after the "member" dimension (unless 
\code{drop_realization_dim=TRUE} is specified).
The ordering of the remaining dimensions in the \code{$data} element of the input object is maintained.
}
\description{
This function implements the RainFARM stochastic precipitation
downscaling method and accepts a CSTools object (an object of the class 
's2dv_cube' as provided by `CST_Load`) as input.
Adapted for climate downscaling and including orographic correction
as described in Terzago et al. 2018.
}
\examples{
#Example 1: using CST_RainFARM for a CSTools object
nf <- 8   # Choose a downscaling by factor 8
exp <- 1 : (2 * 3 * 4 * 8 * 8)
dim(exp) <- c(dataset = 1, member = 2, sdate = 3, ftime = 4, lat = 8, lon = 8)
lon <- seq(10, 13.5, 0.5)
dim(lon) <- c(lon = length(lon))
lat <- seq(40, 43.5, 0.5)
dim(lat) <- c(lat = length(lat))
data <- list(data = exp, lon = lon, lat = lat)
# Create a test array of weights
ww <- array(1., dim = c(8 * nf, 8 * nf))
res <- CST_RainFARM(data, nf, ww, nens=3)
str(res)
#List of 3
# $ data: num [1, 1:2, 1:3, 1:3, 1:4, 1:64, 1:64] 260 553 281 278 143 ...
# $ lon : num [1:64] 9.78 9.84 9.91 9.97 10.03 ...
# $ lat : num [1:64] 39.8 39.8 39.9 40 40 ...
dim(res$data)
# dataset      member realization       sdate       ftime         lat         lon 
#       1           2           3           3           4          64          64

}
\references{
Terzago, S. et al. (2018). NHESS 18(11), 2825-2840.
http://doi.org/10.5194/nhess-18-2825-2018 ;
D'Onofrio et al. (2014), J of Hydrometeorology 15, 830-843; Rebora et. al. (2006), JHM 7, 724.
}
\author{
Jost von Hardenberg - ISAC-CNR, \email{j.vonhardenberg@isac.cnr.it}
}
